from celery.utils.log import get_task_logger
from django.db.models import Count

from wiki.pages.management.commands.tmp_remove_unused_objects import remove_old_revisions
from wiki.pages.models import Revision
from wiki.utils.tasks.base import LockedCallableTask


class RemoveExcessRevisionsTask(LockedCallableTask):
    """
    Удалить лишние (избыточные) ревизии страниц для освобождения места в хранилище.
    """

    name = 'wiki.remove_excess_revisions'
    logger = get_task_logger(__name__)
    time_limit = 60 * 60  # 3600 сек
    lock_name_tpl = 'remove_excess_revisions'
    max_allowable_revision_number = 800

    def run(self, *args, **kwargs):
        revisions = (
            Revision.objects.values('page_id')
            .annotate(rev_num=Count('page_id'))
            .filter(rev_num__gt=self.max_allowable_revision_number)[:100]
        )

        for revision in revisions:
            # у удаленной страницы больше recent_revisions_count ревизий, оставляем только
            # recent_revisions_count самых свежих ревизий, а остальные удаляем
            self.logger.info(
                'Page id={} has {} revisions. Try to remove excess revisions.'.format(
                    revision['page_id'], revision['rev_num']
                )
            )
            remove_old_revisions(page_id=revision['page_id'], recent_revisions_count=self.max_allowable_revision_number)
