"""
При обновлении прав доступа страниц нужно отдать все подстраницы, наследующие права доступа, для индексатора.

@author: vladmos
"""

from celery.utils.log import get_task_logger

from wiki.org import org_ctx
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.models import Page
from wiki.utils import timezone
from wiki.utils.tasks.base import CallableTask

logger = get_task_logger(__name__)


class UpdateClusterForIndexerTask(CallableTask):
    name = 'wiki.update_cluster_for_indexer'
    logger = get_task_logger(__name__)
    time_limit = 60  # 60 сек.

    def run(self, page_id, *args, **kwargs):
        try:
            page = Page.objects.get(id=page_id)
        except Page.DoesNotExist:
            logger.error('Cannot get Page:pk={0}. Maybe it was deleted?'.format(page_id))
            return

        with org_ctx(page.org):
            prefix_to_ignore = '/'  # Заведомо невалидный префикс для супертега

            page_descendants = list(page.descendants)
            descendants_access = get_bulk_raw_access(page_descendants)

            for descendant in page_descendants:
                # подстраницы отсортированы по супертегу, значит, обход в глубину

                if descendant.supertag.startswith(prefix_to_ignore):
                    continue

                try:
                    access = interpret_raw_access(descendants_access[descendant])
                except (KeyError, TypeError):
                    # Если по какой-то причине не удалось получить права подстраницы (например, подстраница была создана
                    # после вызова get_bulk_raw_access, все равно добавим ее к списку на индексацию, так безопаснее.
                    logger.exception('Unexpected error while determining subpages access')
                else:
                    if not access['is_inherited']:
                        # Доступ у этого наследника свой, изменение прав доступа у исходной страницы на эту не повлияло,
                        # дальше вглубь по дереву можно не идти.
                        prefix_to_ignore = descendant.supertag
                        continue

                descendant.modified_at_for_index = timezone.now()
                descendant.save()

            page.modified_at_for_index = timezone.now()
            page.save()
