
from datetime import datetime
from math import ceil

from django import template
from django.contrib.auth import get_user_model
from django.utils.translation import get_language, gettext, ugettext
from wiki.intranet.models import Staff

from wiki.actions import call_action
from wiki.users.logic.settings import get_user_setting
from wiki.utils.timezone import make_naive_current

register = template.Library()


@register.filter
def declination_verb(user):
    """
    User is one of Staff, django.contrib.auth.models.User
    """
    try:
        if isinstance(user, Staff):
            staff = user
        else:
            try:
                staff = user.staff
            except Staff.DoesNotExist:
                return ''
        if get_language()[:2] == 'ru' and getattr(staff, 'gender', 'M') == 'F':
            return 'а'
        return ''

    except Exception:
        return ''


@register.filter
def human_file_size(size):
    sizes = gettext('b|Kb|Mb|Gb|Tb').split('|')
    sizes_len = len(sizes)
    i = 0
    sz = float(size)
    while sz / 1024 > 1 and i + 1 < sizes_len:
        sz /= 1024
        i += 1
    if sz < 100:
        t = 10 if sz > 10 else 100
        sz = ceil(sz * t) / t
    else:
        sz = ceil(sz)
    return str(sz) + ' ' + sizes[i]


@register.filter
def human_datetime(dt):
    """Вернуть полную локализованную дату и время как принято на Вики"""
    if dt is None:
        return ''

    months = ugettext(
        ugettext('januaryz|februaryz|marchz|aprilz|mayz|junez|julyz|augustz|septemberz|octoberz|novemberz|decemberz')
    ).split('|')

    dt = make_naive_current(dt)

    if months and len(months) == 12 and months[0] == 'января':
        # russian
        if not isinstance(dt, datetime):
            return '%d %s %d' % (dt.day, months[dt.month - 1], dt.year)
        return '%d %s %d, %02d:%02d' % (dt.day, months[dt.month - 1], dt.year, dt.hour, dt.minute)
    else:
        # english and other
        return dt.strftime('%H:%M, %B %d, %Y')


@register.filter
def page_url(page):
    """Вернуть абсолютный URL до страницы"""
    return page.url


@register.simple_tag
def render_action(action_name, request, page=None, params=None):
    if params is None:
        params = {}

    action_name = action_name.lower()
    ActionName = action_name.capitalize()

    try:
        module = __import__('wiki.actions.classes.' + action_name, fromlist=[ActionName])
    except ImportError:
        return gettext('Failed to import module %s') % action_name
    klass = getattr(module, ActionName)
    action = klass(params, request)
    if request.method == 'GET':
        return action.get()
    else:
        return action.post()


@register.simple_tag
def show_files(request):
    return call_action(
        'files', request=request, params=dict(nomark=True, is_upload=1), page_path='/' + request.page.supertag
    )


@register.simple_tag
def inflect_user(user, form='subjective'):
    staff = None
    if isinstance(user, Staff):
        staff = user
    elif isinstance(user, get_user_model()):
        staff = user.staff
    if staff is None:
        return '%s (%s)' % (str(user), gettext('not synced with center'))
    return getattr(staff.inflections, form)


@register.filter
def code_theme(user):
    if not isinstance(user, Staff) and not isinstance(user, get_user_model()):
        return 'github'  # for AnonymousUser
    return get_user_setting(user, 'code_theme')
