import itertools
import re

from django.contrib.auth import get_user_model
from django.http import Http404

from wiki.utils.timezone import validate_request_time_str


def split_wiki_name(wiki_name, start=1):
    for i in range(start, len(wiki_name)):
        if wiki_name[i].lower() != wiki_name[i] and wiki_name[i - 1] != ' ':
            return split_wiki_name(wiki_name[0:i] + ' ' + wiki_name[i:], i + 1)
    return wiki_name


def flatten(listOfLists):
    'Flatten one level of nesting'
    return itertools.chain.from_iterable(listOfLists)


regexp_group1 = (
    re.compile(r'\(\((!?[^\/][\.\-\/\w\d:]*)([^\)]*)\)\)'),
    re.compile(r'\[\[(!?[^\/][\.\-\/\w\d:]*)([^\]]*)\]\]'),
)
regexp_group2 = (
    re.compile(r'(["\s\']{1,1})(!\/[\.\-\/\w\d]*)(["\s\']{1,1})'),
    re.compile(r'(["\s\']{1,1})(\.\.\/[\.\-\/\w\d]*)(["\s\']{1,1})'),
)
special_link = re.compile(r'www\.\S*', re.IGNORECASE)


def get_page_author(page, default_user):
    """Вернуть пользователя, от имени которого переформатируем табличый список.

    Может вернуть неизвестно какого пользователя, если у табличного списка не определен last_user

    User, User|None -> User
    """
    if page.last_author:
        return page.last_author
    if default_user:
        return default_user
    return get_user_model().objects.all()[0]


def check_revision(page, rev_time):
    """Вернуть старую версию страницы, если она запрошена

    @type rev_time: str
    @raise Http404
    @return Revision

    """
    try:
        rev_time = validate_request_time_str(rev_time)
    except ValueError:
        # такая ошибка может произойти только если пользователь сам сформировал запрос
        # чего он делать не должен
        raise Http404('Revision time string is badly formatted')
    try:
        return page.revision_set.filter(created_at=rev_time)[0]
    except IndexError:
        raise Http404('Revision not found')
