import os
import tarfile
import time
from io import BytesIO

from wiki.files.models import File
from wiki.pages.models import Page, Revision

DUMP_DIRECTORIES = [
    ('page', Page),
    ('revision', Revision),
    ('file', File),
]


class ReadFileSystem(object):
    """
    Класс, предоставляющий общий интерфейс к работе как с обычными файлами, так и с tar
    """

    def __init__(self, data_path):
        self.data_path = data_path
        self.use_tar = not os.path.isdir(data_path)

    def __enter__(self):
        if self.use_tar:
            self._tar = tarfile.open(self.data_path)
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.use_tar:
            self._tar.close()

    def list_files(self, logger):
        """
        Find all pages data in data_path
        @param logger: script logger
        @rtype: list
        @return: list of serialized pages data filenames
        """
        logger.info('Searching pages backups in "{0}"'.format(self.data_path))

        filenames = []
        if self.use_tar:
            raw_filenames = [t.name for t in self._tar]
            files = dict((s, {}) for s, o in DUMP_DIRECTORIES)
            for filename in raw_filenames:
                for subdirectory, ObjectClass in DUMP_DIRECTORIES:
                    if filename.startswith(subdirectory + '/'):
                        try:
                            object_id = int(os.path.split(filename)[1])
                            files[subdirectory][object_id] = filename
                        except ValueError:
                            logger.warning('wrong filename {0}'.format(filename))

            for subdirectory, ObjectClass in DUMP_DIRECTORIES:
                objects_ids = list(files[subdirectory].keys())
                for chunk_index in range((len(objects_ids) / 1000) + 1):
                    objects_ids_chunk = objects_ids[(chunk_index * 1000) : ((chunk_index + 1) * 1000)]
                    real_ids = ObjectClass.objects.filter(pk__in=objects_ids_chunk).values_list('pk', flat=True)
                    filenames.extend((ObjectClass, id, files[subdirectory][id]) for id in real_ids)

        else:
            filenames = []
            for subdirectory, ObjectClass in DUMP_DIRECTORIES:
                full_subdirectory = os.path.join(self.data_path, subdirectory)
                result = {}
                if os.path.isdir(full_subdirectory):
                    for directory in [
                        d for d in os.listdir(full_subdirectory) if os.path.isdir(os.path.join(full_subdirectory, d))
                    ]:
                        # directory name must be a integer division result from (page.id / FILES_PER_DIR) expression
                        try:
                            int(directory)
                        except ValueError:
                            logger.warning('wrong directory name {0}'.format(os.path.join(subdirectory, directory)))
                            continue
                        else:
                            # get page ids from filenames in subdirectories
                            for fname in os.listdir(os.path.join(full_subdirectory, directory)):
                                full_path = os.path.join(subdirectory, directory, fname)
                                try:
                                    object_id = int(os.path.basename(full_path))
                                except ValueError:
                                    logger.warning('wrong filename {0}'.format(full_path))
                                else:
                                    # ensure unique ids with dict
                                    result[object_id] = full_path

                # get objects' ids
                objects_ids = list(result.keys())

                real_ids = ObjectClass.objects.filter(pk__in=objects_ids).values_list('pk', flat=True)
                filenames.extend((ObjectClass, id, result[id]) for id in real_ids)

        # return filenames
        return filenames

    def read_file(self, file_path):
        if self.use_tar:
            extracted_file = self._tar.extractfile(file_path)
            return extracted_file.read()
        else:
            with open(os.path.join(self.data_path, file_path)) as file:
                return file.read()


class WriteFileSystem(object):
    """
    Класс, предоставляющий общий интерфейс к работе как с обычными файлами, так и с tar
    """

    def __init__(self, data_path, use_tar=False):
        self.data_path = data_path
        self.use_tar = use_tar

    def __enter__(self):
        if self.use_tar:
            self._tar = tarfile.open(self.data_path, 'w:gz')
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.use_tar:
            self._tar.close()

    def add_file(self, filename, file_data):
        if self.use_tar:
            buffer = BytesIO(file_data)
            tar_info = tarfile.TarInfo(filename)
            tar_info.size = len(file_data)
            tar_info.mtime = time.time()
            self._tar.addfile(tar_info, buffer)
        else:
            full_path = os.path.join(self.data_path, filename)
            directory_name = os.path.split(full_path)[0]
            if not os.path.isdir(directory_name):
                os.makedirs(directory_name)
            with open(full_path, 'wb') as f:
                f.write(file_data)
