"""
Set of functions that helps to mark page as deleted
"""
import logging
from time import time

from django.db import IntegrityError
from django.db.transaction import atomic

from wiki.pages.dao.actuality import remove_actuality_mark_links_to
from wiki.pages.models import AbsentPage, Page, PageLink
from wiki.utils import timezone

logger = logging.getLogger(__name__)
# tag renaming template for removed pages
REMOVED_TAG_TEMPLATE = '.deleted.{timestamp}/{tag}'


@atomic
def delete_page(page):
    """
    Пометить страницу и все редиректы на нее как удаленные.
    Также создает AbsentPage объекты для всех ссылок на удаленную страницу и
    удаляет PageLink объекты из базы.

    @type page: Page
    """
    # пометить редиректы удаленными
    _remove_redirects(page)

    # удалить ссылки
    _unlink_page(page)

    # удалить ссылки с устаревших страниц на эту как более актуальную
    remove_actuality_mark_links_to(page)

    # пометить страницу как удаленную
    _mark_page_removed(page)


def _mark_page_removed(page):
    """
    Выставить странице поле status = 0 (страница удалена),
    пересохранить ее с префиксом перед тэгом и супертэгом.

    @type page: Page
    @param page: удаляемая страница
    """
    page.status = 0

    timestamp = str(time())
    page.tag = REMOVED_TAG_TEMPLATE.format(timestamp=timestamp, tag=page.tag)[:255]
    page.supertag = REMOVED_TAG_TEMPLATE.format(timestamp=timestamp, tag=page.supertag)[:255]
    page.modified_at = timezone.now()

    page.save()


def _unlink_page(page):
    """
    Создать AbsentPage объекты для всех ссылок на страницу и
    удалить PageLink объекты из базы.

    @type page: Page
    """
    page_links = PageLink.objects.filter(to_page=page)
    for link in page_links:
        try:
            AbsentPage(from_page=link.from_page, to_tag=page.tag, to_supertag=page.supertag).save()
        except Page.DoesNotExist:
            link.delete()
        except IntegrityError:
            logger.exception('AbsentPage integrity error')
    page_links.delete()


def _remove_redirects(page):
    """
    Пометить все редиректы на страницу как удаленные

    @type page: Page
    """
    for redirect in Page.active.filter(redirects_to=page):
        if redirect.id != page.id:
            delete_page(redirect)
