import logging

from django.conf import settings
from django.http import HttpRequest

from wiki.intranet.models import Staff
from wiki.legacy.langs import get_default_lang, get_supported_lang
from wiki.org import get_org
from wiki.pages.models import Page
from wiki.utils.supertag import translit

logger = logging.getLogger(__name__)


def prepare_request(page_or_url=None, user=None):
    """Сформировать объект HttpRequest

    Используйте, если у вас в контексте нет HTTP запроса
    """
    request = HttpRequest()
    request.META['SERVER_NAME'] = settings.NGINX_HOST

    if user:
        request.user = user
    elif hasattr(page_or_url, 'last_author'):
        # Get user from page.last_author property.
        request.user = page_or_url.last_author
    else:
        raise ValueError('Cannot detect user from arguments. Pass "user" argument, please')

    if hasattr(page_or_url, 'pk'):
        request.page = page_or_url
    else:
        try:
            request.page = Page.objects.get(supertag=translit(page_or_url), org=get_org())
            request.tag = request.page.tag
        except Page.DoesNotExist:
            pass

    profile = None
    if request.user and not request.user.is_anonymous:
        try:
            profile = request.user.staff
        except Staff.DoesNotExist:
            logger.warning(
                'Could not find profile for auth.User:pk="%s","%s", page="%s"',
                request.user.id,
                request.user.username,
                page_or_url,
            )

    request.session = {}

    if profile:
        request.LANGUAGE_CODE = get_supported_lang(profile.lang_ui)
    else:
        request.LANGUAGE_CODE = get_default_lang()

    setattr(request, 'from_yandex_server', False)

    return request
