"""
Set of functions that helps to restore previously deleted pages
"""
import logging
import re

from django.db import IntegrityError
from django.db.transaction import atomic
from psycopg2 import errorcodes as pg_errorcodes

from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.org import get_org
from wiki.pages.models import AbsentPage, Page, PageLink

logger = logging.getLogger(__name__)
# deleted page supertag pattern
prefix_pattern = re.compile(r'\.deleted\.\d+?\.?\d+?/')


class RestorePageError(InvalidDataSentError):
    pass


@atomic
def resurrect_page(page):
    """
    Make once deleted page active again

    @type page: Page
    @param page: deleted page
    """
    # we'll only resurrect deleted pages
    if page.is_active:
        raise RestorePageError('Page "{0}" is active already'.format(page.supertag))

    # restore page status, tag and supertag
    _restore_page_state(page)

    # restore links to page
    _restore_page_links(page)

    # restore redirects to page
    _restore_page_redirects(page)


def _restore_page_state(page):
    """
    Set page status = 1 (page is active),
    strip page tag and supertag from deleted prefix

    @type page: Page
    @param page: deleted page
    """
    page.status = 1
    for field_name in ('tag', 'supertag'):
        value = getattr(page, field_name)
        if not value:
            continue
        setattr(page, field_name, restore_deleted_slug(value))

    try:
        page.save()
    except IntegrityError as exc:
        if exc.__cause__.pgcode == pg_errorcodes.UNIQUE_VIOLATION:
            raise RestorePageError(f'Page "{page.supertag}" already exists')


def restore_deleted_slug(slug):
    """
    Восстановить тег или супертег удаленной страницы.

    @type address: unicode|None
    @param address: тег или супертег
    @rtype: unicode
    """
    if not prefix_pattern.match(slug):
        return slug
    return slug.split('/', 1)[1]


def _restore_page_redirects(page):
    """
    Make all page redirects active again

    @type page: Page
    """
    for redirect in Page.objects.filter(redirects_to=page):
        if redirect.id != page.id and redirect.status == 0:
            resurrect_page(redirect)


def _restore_page_links(page):
    """
    Delete all AbsentPage objects linked to page and restore PageLinks from them.

    @type page: Page
    """
    link_stubs = AbsentPage.objects.filter(to_supertag=page.supertag, from_page__org=get_org())
    for absent_link in link_stubs:
        try:
            PageLink(from_page=absent_link.from_page, to_page=page).save()
        except Page.DoesNotExist:
            absent_link.delete()
        except IntegrityError:
            logger.exception('PageLink integrity error')

    link_stubs.delete()
