import json
import logging

import django.http as http
from django.conf import settings
from django.utils.translation import ugettext

from wiki.api_core.errors.permissions import UserHasNoAccess
from wiki.api_core.errors.rest_api_error import ResourceIsMissing, UnhandledException
from wiki.api_core.framework import WikiAPIView
from wiki.arc_compat import read_asset

_errors = {
    404: (ResourceIsMissing, http.HttpResponseNotFound),
    403: (UserHasNoAccess, http.HttpResponseForbidden),
    500: (UnhandledException, http.HttpResponseServerError),
}

_favicon_image_data = None


def response_with_error(status, title):
    # Django требует возвращать свои реализации HttpResponse в этих view.
    # А json с описанием ошибки берем из нашего WikiAPIView.

    framework_exc_class, django_resp_class = _errors[status]

    exc = framework_exc_class(title)
    view = WikiAPIView()
    response = view.handle_exception(exc)
    data = response.data

    return django_resp_class(json.dumps(data), content_type='application/json')


def server_error(request, exception=None):
    logging.exception(f'Unhandled exception: {exception}')
    return response_with_error(500, ugettext('Server error'))


def page_not_found(request, exception=None):
    return response_with_error(404, ugettext('Page not found'))


def page_not_allowed(request, exception=None):
    return response_with_error(403, ugettext('You are not allowed to view this page'))


def read_only(request):
    return response_with_error(500, ugettext('wiki:ReadOnlyWarning'))


def tilde_redirect_to_user(request, after_tilde):
    from wiki.api.views.resolve_tilde import resolve_tilde

    url = resolve_tilde(request, after_tilde)
    return http.HttpResponseRedirect(url)


def favicon(request):
    global _favicon_image_data
    if not _favicon_image_data:
        _favicon_image_data = read_asset(settings.FAVICON_PATH)
    return http.HttpResponse(_favicon_image_data, content_type='image/png')
