"""
    Описывает функции для получения и правки заметок.
"""
import logging

from django.conf import settings
from django.db import IntegrityError, transaction
from django.utils.translation import activate, get_language, ugettext

from wiki.intranet.models import Staff
from wiki.org import get_org, get_org_lang, org_staff
from wiki.pages.api import create_event, save_page
from wiki.pages.logic.clone import clone_page
from wiki.pages.logic.etalon import get_etalon_pages
from wiki.pages.models import Page

logger = logging.getLogger(__name__)


def supertag_of_user_notes(user_cluster):
    """Вернуть супертег для кластера заметок

    Page -> string

    """
    return '{0}/notes'.format(
        user_cluster.supertag,
    )


def previous_notes(user_cluster, count=3):
    """Выбрать несколько самых поздних заметок пользователя

    Page, int -> QuerySet
    """
    return Page.active.filter(supertag__startswith=supertag_of_user_notes(user_cluster) + '/', org=get_org()).order_by(
        '-created_at'
    )[:count]


def get_notes_cluster(user_cluster):
    """Вернуть кластер для пользовательских заметок.

    Page -> Page
    """
    supertag = supertag_of_user_notes(user_cluster)
    try:
        return Page.objects.get(supertag=supertag, org=get_org())
    except Page.DoesNotExist:
        return create_notes_cluster(user_cluster)


@transaction.atomic
def create_notes_cluster(user_cluster):
    """Создать кластер для пользовательских заметок.

    Page -> Page

    """
    # TODO: решить проблему циклических импортов. - ух!
    from wiki.access import set_access, TYPES, NoChanges

    if settings.IS_BUSINESS:
        pages = get_etalon_pages(['personalpage/notes'], get_org_lang())
        if len(pages) == 1:
            page = clone_page(
                page=pages[0],
                new_tag=supertag_of_user_notes(user_cluster),
                authors=user_cluster.get_authors(),
                last_author=user_cluster.get_authors().first(),
                is_autogenerated=True,
            )

            try:
                set_access(
                    page,
                    TYPES.OWNER,
                    author_of_changes=user_cluster.get_authors().first(),
                    send_notification_signals=False,
                )
            except NoChanges:
                pass

            return page

    cur_language = get_language()
    try:
        if hasattr(user_cluster.get_authors().first(), 'staff') and user_cluster.get_authors().first().staff.lang_ui:
            language = user_cluster.get_authors().first().staff.lang_ui
        else:
            try:
                staff = org_staff().get(login=user_cluster.get_authors().first().username)
                language = staff.lang_ui
            except (Staff.DoesNotExist, Staff.MultipleObjectsReturned):
                language = None

        if language:
            activate(language)

        page, rev, _ = save_page(
            supertag_of_user_notes(user_cluster),
            unicode_text='{{ tree }}',
            title=ugettext('base:Personal notes'),
            user=user_cluster.get_authors().first(),
            authors=user_cluster.get_authors(),
        )
        page.is_autogenerated = True
        page.save()

        create_event(True, page, rev, notify=False)
    except IntegrityError as exc:
        logger.warning('Can\'t create user\'s \'personalpage/notes\' page: %s', exc)
        return None
    finally:
        activate(cur_language)

    try:
        set_access(
            page, TYPES.OWNER, author_of_changes=user_cluster.get_authors().first(), send_notification_signals=False
        )
    except NoChanges:
        pass

    return page
