import logging
import re

from django.conf import settings
from django.core.cache import caches
from django.db import IntegrityError, transaction
from django.template.loader import render_to_string
from ylog.context import log_context

from wiki.intranet.models import Staff
from wiki.org import get_org, get_org_lang, get_user_orgs, org_ctx, org_staff
from wiki.pages.api import create_event, save_page
from wiki.pages.logic.clone import clone_page
from wiki.pages.logic.etalon import get_etalon_pages
from wiki.pages.models import Page, RedirectLoopException
from wiki.personalisation.quick_notes import get_notes_cluster
from wiki.utils import timezone
from wiki.utils.context import get_org_context
from wiki.utils.supertag import translit

__cache = None

logger = logging.getLogger(__name__)


def cache():
    """
    Ленивая инициализация кеша."""
    global __cache
    if __cache is None:
        __cache = caches['wiki_state']
    return __cache


def create_personal_page(django_user):
    """
    Create (or find) a personal page for a user. Usually this function is
    invoked from signal receiver src/wiki/pages/signals/create_user_cluster.py

    If login has "_" or "." in it, page's supertag won't have these symbols
    after wiki transliteration. To make life of such users easier we also
    create a redirection from /users/LOGIN to transliterated supertag.

    Pages created for user dorothy:
        /users/dorothy

    Pages created for user tin_man:
        /users/tinman
        /users/tin_man -> /users/tinman
    """
    expected_tag = 'users/' + django_user.username
    tag = translit(expected_tag)
    user_cluster = None
    for org in get_user_orgs(django_user):
        with org_ctx(org):
            with log_context(**get_org_context()):
                user_cluster = get_or_create_page(django_user, tag)
                if expected_tag != tag:
                    get_or_create_redirection(django_user, expected_tag, user_cluster)
                get_notes_cluster(user_cluster)
    return user_cluster


def get_user_personal_cluster_slug(django_user) -> str:
    return translit('users/' + django_user.username)


def has_user_personal_cluster(django_user):
    tag = get_user_personal_cluster_slug(django_user)
    return Page.objects.filter(supertag=tag, org=get_org()).exists()


def get_or_create_page(django_user, tag):
    try:
        page = Page.objects.get(supertag=tag, org=get_org())
    except Page.DoesNotExist:
        try:
            page = create_page(django_user, tag)
            logger.info('Page {0!r} created'.format(tag))
        except IntegrityError as exc:
            logger.warning('Can\'t create user\'s personal page: %s', exc)
            # страница уже была создана на другом воркере при обработке параллельного запроса
            page = Page.objects.get(supertag=tag, org=get_org())
    else:
        logger.info('Page {0!r} already exists'.format(tag))
    return page


@transaction.atomic
def create_page(django_user, tag):
    if settings.IS_BUSINESS:
        pages = get_etalon_pages(['personalpage'], get_org_lang())
        if pages:
            page = clone_page(
                page=pages[0],
                new_tag=tag,
                authors=[django_user],
                last_author=django_user,
                title='{0} {1} ({2})'.format(django_user.first_name, django_user.last_name, django_user.username),
                is_autogenerated=True,
                _should_skip_reserved_protection=True,
                with_new_wf=True,
            )
            page.save()
            page.authors.add(django_user)
            return page

    page = Page(
        tag=tag,
        supertag=tag,
        owner=django_user,
        org=get_org(),
        is_autogenerated=True,
        with_new_wf=True,
    )
    page._should_skip_reserved_protection = True

    page_body = render_to_string(settings.PERSONAL_PAGE_TEMPLATE, {'login': django_user.username})

    page, rev, _ = save_page(
        page,
        unicode_text=page_body,
        title='{0} {1} ({2})'.format(django_user.first_name, django_user.last_name, django_user.username),
        user=django_user,
    )
    page.authors.add(django_user)

    create_event(True, page, rev, notify=False)

    return page


def get_or_create_redirection(django_user, from_tag, to_page):
    if Page.objects.filter(supertag=from_tag, org=get_org()).exists():
        return
    from_page = Page(
        owner=django_user,
        last_author=django_user,
        tag=from_tag,
        supertag=from_tag,
        modified_at=timezone.now(),
        org=get_org(),
        with_new_wf=True,
    )
    from_page._should_skip_reserved_protection = True
    from_page.redirects_to = to_page
    from_page.save()
    from_page.authors.add(django_user)


class NoPersonalCluster(Exception):
    pass


RE = re.compile(r'^/*~(?=(/|$))')


def expand_user_cluster_alias(path: str, user):
    # Заменяем /~/anything -> user/login/anything
    # Игнорируем всю логику про старые вида персональных кластеров
    # - для всех пользователей уже созданы страницы вида users/{0}
    # - users/{0} при наличии отдается в первую очередь из personal_cluster
    # - единственное - редиректы или цепочки редиректов, но они вернутся в сериализации

    if not path or not RE.match(path):
        return path

    home = 'users/{0}'.format(translit(user.username))
    return RE.sub(home, path)


def personal_cluster(user):
    """Вернуть страницу-корень личного кластера пользователя.

    Проверить по новому адресу /users/логин/, потом по старому, /wiki-имя/.

    django.contrib.auth.User -> Page

    @raise NoPersonalCluster если личного кластера нет.

    """
    try:
        cluster = Page.objects.get(supertag='users/{0}'.format(user.username), org=get_org())
        return cluster.redirect_target()
    except RedirectLoopException:
        return cluster
    except Page.DoesNotExist:
        pass

    try:
        staff = user.staff
    except Staff.DoesNotExist:
        logger.warning('Django user "%s" does not have staff model', user.username)
        raise

    try:
        cluster = Page.objects.get(supertag=translit(staff.wiki_name), org=get_org())
    except Page.DoesNotExist:
        logger.warning('No personal cluster of auth.User name="%s"', user.username)
        raise NoPersonalCluster()

    try:
        cluster = cluster.redirect_target()
    except RedirectLoopException:
        logger.exception('Page:pk="%s" is in a redirect loop', cluster.id)
    return cluster


def _get_ucluster_tags():
    """
    Вернуть set тэгов старых пользовательских кластеров

    По историческим причинам у нас 2 вида старых пользовательских кластеров:
     - tag вида {login} (пример - vizo, vasiliy )
     - tag вида {staff.wiki_name} (пример - ЖаннаКруглова, VitalyLysenkov)

    @rtype: set
    """
    cache_record_key = 'old_ucluster_tags'
    # пробуем получить список возможных тэгов старых кластеров из кэша
    _old_ucluster_tags = cache().get(cache_record_key)

    if not _old_ucluster_tags:
        # генерируем новый список возможных тэгов старых кластеров

        # 1й вид кластеров - тэги совпадают с wiki_name пользователя на стаффе
        wiki_names = (
            org_staff().filter(wiki_name__isnull=False).exclude(wiki_name='').values_list('wiki_name', flat=True)
        )

        # 2й вид кластеров - тэги совпадают с логином пользователя на стаффе
        logins = org_staff().filter(login__isnull=False).exclude(login='').values_list('login', flat=True)

        _old_ucluster_tags = [translit(wn) for wn in wiki_names if wn] + list(logins)
        # запихиваем список возможных тэгов старых кластеров в кэш на сутки
        cache().set(cache_record_key, _old_ucluster_tags, timeout=24 * 60 * 60)

    return set(_old_ucluster_tags)


def is_in_old_ucluster(page):
    """
    Возвращает, находится ли страница в старом кластере пользователя

    По историческим причинам у нас 2 вида старых пользовательских кластеров:
     - tag вида {login} (пример - vizo, vasiliy )
     - tag вида {staff.wiki_name} (пример - ЖаннаКруглова, VitalyLysenkov)

    Внимание - используется именно tag страницы, т.к. для сравнения с supertag надо производить доп.
    телодвижения (переводить имена в частности)

    К сожалению, не все они редиректы - особенно 2я группа.

    @type page: Page
    @param page: вики-страница
    @rtype: bool
    """
    # список возможных тэгов старых кластеров
    _old_ucluster_tags = _get_ucluster_tags()

    # page.tag может быть не юникод, в отличие от членов сета _old_ucluster_tags
    # потому без приведения к юникоду здесь сравнение не пройдет
    # (т.к. объект хешируется для проверки на членство в сете)
    if translit(str(page.tag.split('/', 1)[0])) in _old_ucluster_tags:
        return True

    return False


def is_in_user_cluster(page):
    """
    Возвращает, находится ли страница в кластере пользователя

    По историческим причинам у нас 3 вида пользовательских кластеров:
     - супертэг вида users/{login} (современный кластер - users/chapson, users/thasonic)
     - супертэг вида {login} (старый кластер - vizo, vasiliy )
     - tag вида {staff.wiki_name} (еще один старый кластер - ЖаннаКруглова, VitalyLysenkov)

    @type page: Page
    @param page: вики-страница
    @rtype: bool
    """
    if page.supertag.startswith('users/'):
        # новый кластер пользователя
        return True
    elif is_in_old_ucluster(page):
        return True

    return False
