
"""
Ping management command.
Prints '0;OK' if all ok.
Prints '2;{error description}' in case of error.
"""
from requests import get

from django.conf import settings
from django.core.management.base import BaseCommand
from django_replicated.dbchecker import db_is_alive


PING_URL = 'https://127.0.0.1:80/ping_database'

# all green
OK = '0;OK'


class PingException(Exception):
    pass


class WrongStatus(PingException):

    """
    wrong http status code in response
    """

    def __str__(self):
        return 'Status code != 200, it\'s {0}!'.format(self.args[0])


def resolve_aliases(aliases):
    """
    Превратить алиасы в имена хостов.

    @type aliases: list
    @rtype: list
    """
    return [settings.DATABASES[alias]['HOST'] for alias in aliases]


class DatabaseError(PingException):

    """
    Некоторые из инстансов базы недоступны.
    """

    aliases = []

    def __init__(self, aliases, *args, **kwargs):
        self.aliases = aliases
        super(DatabaseError, self).__init__(*args, **kwargs)

    def __str__(self):
        return 'Hosts "{0}" are not available.'.format(','.join(resolve_aliases(self.aliases)))


class CriticalDatabaseError(DatabaseError):
    def __str__(self):
        return 'No database instances available: "{0}"!'.format(','.join(resolve_aliases(self.aliases)))


def validate(response):
    """
    Validate ping handle response:
        - http status must be 200

    @type response: requests.Response
    @param response: ping handle http response
    @raise WrongStatus
    """
    if response.status_code != 200:
        raise WrongStatus(response.status_code)


def check_databases(aliases):
    """
    Бросить исключение, если некоторые из инстансов базы недоступны.

    @type aliases: list
    @param aliases: список имен алиасов из settings.DATABASES
    @rtype: None
    """
    unavailable_aliases = []

    for alias in aliases:
        if not db_is_alive(alias, number_of_tries=3):
            unavailable_aliases.append(alias)

    if unavailable_aliases:
        if len(aliases) == len(unavailable_aliases):
            # все инстансы базы недоступны
            raise CriticalDatabaseError(unavailable_aliases)
        raise DatabaseError(unavailable_aliases)


def format_error(message):
    return '2;{0}'.format(message)


class Command(BaseCommand):
    help = 'Ping service and return status in monrun-compliant format'
    requires_system_checks = False

    def handle(self, *args, **options):
        try:
            headers = {'Host': settings.API_WIKI_HOST}
            response = get(PING_URL, headers=headers, timeout=25, verify=False)
            check_databases(list(settings.DATABASES.keys()))
            # validate должно быть всегда последней проверкой.
            validate(response)
        except DatabaseError as exc:
            print(format_error('Database error:"{0}"'.format(exc)))

        except PingException as exc:
            print(format_error(repr(exc)))

        except Exception as exc:
            print(format_error('Unhandled exception:"{0}"'.format(repr(exc))))

        else:
            print(OK)
