from django.db.models import Q
from pathlib import PurePath
from itertools import chain
from typing import List, Iterable, Set, Optional

from django.contrib.auth import get_user_model
from django.db import IntegrityError, transaction
from wiki.acl.check_access import Action, _check_access
from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.exceptions import AlreadyExists
from wiki.pages.models import Page
from wiki.subscriptions.exceptions import SubscribeWithoutAccess
from wiki.subscriptions.models import Subscription, SubscriptionType
from wiki.sync.connect.base_organization import as_base_organization

User = get_user_model()

# Ручка получения подписок отдает и старые и новые подписки
# чтобы их различить, для старых подписок будем увеличивать
# id на число, большее максимального на текущий момент в базе
# примерно в два раза
# Нужно будет потом удалить вместе со старыми подписками
WATCHES_ID_INCREMENT = 50_000_000


def get_parents(slug: str) -> Set[str]:
    return {str(path) for path in PurePath(slug).parents} - {'.', '/'}


def filter_subscriptions_by_exclude(subscriptions: Iterable[Subscription], slug: str) -> List[Subscription]:
    parents = get_parents(slug)
    return [subscription for subscription in subscriptions if parents.isdisjoint(subscription.excludes)]


def filter_subscriptions_by_exclude_batch(subscriptions: Iterable[Subscription], slugs: List[str]) -> List[Subscription]:
    parents = set(chain.from_iterable(get_parents(slug) for slug in slugs))
    return [subscription for subscription in subscriptions if parents.isdisjoint(subscription.excludes)]


def assert_no_subscriptions_to_cluster_above(users: Iterable[User], page: Page):
    if get_subscriptions_to_cluster_above(users=users, page=page):
        raise AlreadyExists()


def get_subscriptions_to_cluster_above(users: Iterable[User], page: Page) -> List[Subscription]:
    subscriptions_above = Subscription.objects.select_related('page').filter(
        user__in=users,
        page__org=page.org,
        is_cluster=True,
        page__supertag__in=get_parents(page.slug),
    )
    return filter_subscriptions_by_exclude(subscriptions_above, slug=page.slug)


def get_subscriptions_to_cluster_above_batch(user: User, pages: Iterable[Page]) -> List[Subscription]:
    parents = set(chain.from_iterable(get_parents(page.slug) for page in pages))
    orgs = {page.org for page in pages}
    if len(orgs) != 1:
        return []

    subscriptions_above = Subscription.objects.select_related('page').filter(
        user=user,
        page__org=orgs.pop(),
        is_cluster=True,
        page__supertag__in=parents
    )
    return filter_subscriptions_by_exclude_batch(subscriptions_above, slugs=[page.slug for page in pages])


def get_user_page_subscriptions(user: User, page: Page) -> Optional[Subscription]:
    for subscr in Subscription.objects.filter(user=user, page=page):
        return subscr
    for subscr in get_subscriptions_to_cluster_above(users=[user], page=page):
        return subscr
    return None


def is_subscribed_to_page(user: User, page: Page):
    return bool(get_user_page_subscriptions(user, page))


def create_subscription(
    user: User, page: Page, type_: SubscriptionType = SubscriptionType.MY, is_cluster: bool = False
) -> Subscription:
    return create_subscriptions(users=[user], page=page, type_=type_, is_cluster=is_cluster)[0]


def create_subscriptions(
    users: Iterable[User], page: Page, type_: SubscriptionType, is_cluster: bool
) -> List[Subscription]:
    for user in users:
        if not _check_access(user, organization=as_base_organization(page.org), slug='', page=page, op=Action.READ):
            raise SubscribeWithoutAccess(details={'user_identity': UserIdentity.from_user(user).dict()})

    assert_no_subscriptions_to_cluster_above(users=users, page=page)

    with transaction.atomic():
        remove_current_subscriptions(users=users, page=page)
        if is_cluster:
            remove_nested_subscriptions(users=users, page=page)

        try:
            models = [Subscription(user=user, page=page, type=type_, is_cluster=is_cluster) for user in users]
            subscriptions = Subscription.objects.bulk_create(models)
        except IntegrityError:
            raise AlreadyExists()

    return subscriptions


def remove_current_subscriptions(users: Iterable[User], page: Page):
    Subscription.objects.filter(user__in=users, page=page).delete()


def remove_nested_subscriptions(users: Iterable[User], page: Page):
    Subscription.objects.filter(page__supertag__startswith=page.slug + '/', user__in=users, page__org=page.org).delete()


def get_page_subscriptions(page: Page, user_filter: str = None):
    # https://docs.djangoproject.com/en/2.2/ref/models/querysets/#in
    subq = Page.objects.filter(org=page.org, supertag__in=get_parents(page.slug)).values_list('id', flat=True).all()

    queryset = Subscription.objects.select_related('page').filter(Q(page__in=subq, is_cluster=True) | Q(page=page))

    if user_filter:
        query_user = (
            Q(user__username__istartswith=user_filter)
            | Q(user__staff__first_name__istartswith=user_filter)
            | Q(user__staff__last_name__istartswith=user_filter)
        )
        queryset = queryset.select_related('user__staff').filter(query_user)

    subscriptions = filter_subscriptions_by_exclude(queryset, slug=page.slug)
    return subscriptions
