from datetime import datetime
from functools import reduce
from operator import or_
from typing import List, Generator, Any, Dict, Set, Tuple, Iterable

from pydantic import BaseModel, Field

from django.db import transaction, IntegrityError
from django.db.models import Q

from wiki.api_v2.exceptions import BadRequest, AlreadyExists
from wiki.pages.access import get_bulk_access_status, ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED
from wiki.pages.models import PageWatch, Page
from wiki.subscriptions.models import Subscription, SubscriptionType
from wiki.sync.connect.models import Organization
from wiki.sync.connect.org_ctx import org_ctx
from wiki.users.logic.settings import set_user_setting
from wiki.users.models import User


MAX_COUNT_PAGE_TO_CHECK_ACCESS = 50_000


class SubscriptionSchema(BaseModel):
    supertag: str
    is_cluster: bool
    exclude: List[str] = Field(default_factory=list)
    created_at: datetime


def candidates(path: str) -> Generator[str, Any, None]:
    for i, ch in enumerate(path):
        if ch == '/':
            yield path[:i]


def collapse_to_clusters(slugs: Iterable[str]) -> Dict[str, Set[str]]:
    clusters = {}
    for slug in sorted(slugs):
        if cluster := next((part for part in candidates(slug) if part in clusters), None):
            clusters[cluster].add(slug)
        else:
            clusters[slug] = set()
    return clusters


def filter_by_access(slugs: Set[str], user: User, org: Organization) -> Set[str]:
    if len(slugs) > MAX_COUNT_PAGE_TO_CHECK_ACCESS:
        raise BadRequest(
            'Failure. Since it is required to check the access of '
            f'{len(slugs)} > {MAX_COUNT_PAGE_TO_CHECK_ACCESS} pages'
        )

    with org_ctx(org):
        access = get_bulk_access_status(tags=slugs, user=user)

    valid_access = frozenset([ACCESS_COMMON, ACCESS_RESTRICTED, ACCESS_UNLIMITED])
    return {slug for slug in slugs if access[slug] in valid_access}


def get_page_watches(username: str, org: Organization) -> Tuple[Set[Tuple[str, str]], Set[Tuple[str, str]]]:
    page_watches = list(
        PageWatch.objects.select_related('page')
        .filter(user=username, page__status=1, page__redirects_to__isnull=True, page__org=org)
        .values_list('page__supertag', 'is_cluster', 'created_at')
    )
    ones = {(slug, created_at) for slug, is_cluster, created_at in page_watches if not is_cluster}
    clusters = {(slug, created_at) for slug, is_cluster, created_at in page_watches if is_cluster}
    return ones, clusters


def get_all_pages(sub_pages: Set[str], org: Organization) -> Set[str]:
    clusters = collapse_to_clusters(sub_pages).keys()  # чтобы получить, только те ветки, где есть подписки
    query = reduce(or_, (Q(supertag__startswith=tag + '/') for tag in clusters))
    all_sub_pages = Page.objects.filter(query, redirects_to__isnull=True, org=org).values_list('supertag', flat=True)
    return set(all_sub_pages)


def make_cluster_subscriptions(
    page_watches: Set[Tuple[str, str]],
    pages: Iterable[str],
    is_create_new: bool = True,
) -> Tuple[List[SubscriptionSchema], List[str]]:

    subscriptions, exclude = [], []
    watches = dict(page_watches)
    watches_slugs = set(watches)
    for slug, cluster_pages in collapse_to_clusters(pages).items():
        if slug in watches:
            subscr_exclude = []

            if not cluster_pages.issubset(page_watches):  # не все подстраницы есть в подписках
                other_subscr, subscr_exclude = make_cluster_subscriptions(
                    page_watches, pages=cluster_pages, is_create_new=False
                )
                subscriptions.extend(other_subscr)

            if is_create_new:
                subscr = SubscriptionSchema(
                    supertag=slug, exclude=subscr_exclude, is_cluster=True, created_at=watches[slug]
                )
                subscriptions.append(subscr)
            else:
                exclude.extend(subscr_exclude)
        else:
            exclude.append(slug)
            if cluster_pages & watches_slugs:
                other_subscr, _ = make_cluster_subscriptions(page_watches, cluster_pages)
                subscriptions.extend(other_subscr)

    return subscriptions, exclude


@transaction.atomic()
def create_model_subscriptions(clusters_subscriptions: List[SubscriptionSchema], user: User, org) -> List[Subscription]:
    supertags = [cluster.supertag for cluster in clusters_subscriptions]
    pages_ids = dict(Page.objects.filter(supertag__in=supertags, org=org).values_list('supertag', 'id'))

    subscriptions = []
    for cluster in clusters_subscriptions:
        if cluster.supertag in pages_ids:  # защита, если страница была удалена в процессе
            subscr = Subscription(
                user=user,
                page_id=pages_ids[cluster.supertag],
                is_cluster=cluster.is_cluster,
                exclude={'exclude': sorted(cluster.exclude)},
                type=SubscriptionType.MY,
                created_at=cluster.created_at,
            )
            subscriptions.append(subscr)

    try:
        created_subscriptions = Subscription.objects.bulk_create(subscriptions)
    except IntegrityError:
        raise AlreadyExists()

    return created_subscriptions


def assert_no_subscription_or_delete(user: User, org: Organization, force=False):
    subscriptions = Subscription.objects.select_related('page').filter(user=user, page__org=org)
    if subscriptions.exists():
        if not force:
            raise AlreadyExists('The user has already migrated')
        subscriptions.delete()


def migrate_page_watches_to_subscriptions(user: User, org: Organization, force=False):
    assert_no_subscription_or_delete(user=user, org=org, force=force)

    ones, sub_pages = get_page_watches(username=user.username, org=org)
    subscriptions = [
        SubscriptionSchema(supertag=slug, is_cluster=False, created_at=created_at) for slug, created_at in ones
    ]

    if sub_pages:
        sub_pages_slugs = {slug for slug, _ in sub_pages}
        ones_slugs = {slug for slug, _ in ones}

        all_pages = get_all_pages(sub_pages=sub_pages_slugs, org=org)
        accessible_pages = filter_by_access(slugs=all_pages - sub_pages_slugs - ones_slugs, user=user, org=org)
        accessible_pages |= ones_slugs | sub_pages_slugs  # suppose that sub_pages already have access

        clusters_subscriptions, _ = make_cluster_subscriptions(sub_pages, accessible_pages)
        subscriptions.extend(clusters_subscriptions)
    create_model_subscriptions(subscriptions, user=user, org=org)

    with org_ctx(org):
        set_user_setting(user, 'new_subscriptions', True)
