import json
from enum import Enum
from typing import List, Iterable

from django.contrib.auth import get_user_model
from django.db import transaction
from django.db.models import Q
from ninja import Schema
from pydantic import Field
from pydantic import root_validator
from wiki.api_v2.exceptions import BadRequest

from wiki.pages.models import Page
from wiki.support_tools.consts import SupportOperation
from wiki.support_tools.models import log_operation
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.org_identity import OrganizationIdentity
from wiki.utils.django_redlock.redlock import RedisLock
from wiki.utils.supertag import normalize_supertag

User = get_user_model()


class Mode(str, Enum):
    REPLACE = 'replace'  # update only those pages, where there are authors to delete
    MODIFY = 'modify'  # add and remove, where possible


class ChangeAuthorsRequest(Schema):
    slug: str = Field(description='slug кластера')
    org: OrganizationIdentity = Field(default_factory=OrganizationIdentity, description='Организация')

    dry_run: bool = Field(default=True, description='вывести список изменений, но не применять их')
    remove_usernames: List[str] = Field(default_factory=list, description='список логинов пользователей для удаления')
    add_usernames: List[str] = Field(default_factory=list, description='список логинов пользователей для добавления')
    mode: Mode = Field(
        default=Mode.MODIFY, description='replace - обновляются только в те страницы, где есть люди на удаление'
    )

    @root_validator
    def validation(cls, data):
        assert data.get('remove_usernames') or data.get(
            'add_usernames'
        ), 'One of the fields is required: "remove_usernames" or "add_usernames"'
        if data.get('mode') == Mode.REPLACE:
            assert data.get('remove_usernames'), 'Field "remove_usernames" is required, when mode == "replace"'
        return data


class ChangeAuthorsResponse(Schema):
    changes: List[str]
    dry_run: bool


def get_users(usernames: List[str], org: BaseOrganization) -> List[User]:
    if not usernames:
        return []

    query = org.get_users().filter(username__in=usernames)
    users = list(query)

    if len(users) != len(usernames):
        lost_usernames = set(usernames) - {u.username for u in users}
        raise BadRequest(f'Not all usernames are found - [{", ".join(lost_usernames)}]')
    return users


def get_pages(data: ChangeAuthorsRequest, org: BaseOrganization) -> Iterable[Page]:
    page_filter = Q(supertag__startswith=normalize_supertag(data.slug) + '/') | Q(supertag=data.slug)

    if data.mode == Mode.REPLACE:
        page_filter &= Q(authors__username__in=data.remove_usernames)

    return org.get_pages().filter(page_filter)


def change_authors_view(request, data: ChangeAuthorsRequest) -> ChangeAuthorsResponse:
    """
    Добавить/удалить авторов в страницах подкластера
    Запрос может сделать только пользователь из IDM группы Support

    Не допускается удалять всех авторов со страницы
    Результаты логируются в таблицу SupportLog
    """

    org = data.org.to_organization()

    with RedisLock(name=f'ChangeAuthors: {data.slug}'), transaction.atomic():
        add_users = get_users(usernames=data.add_usernames, org=org)
        remove_users = get_users(usernames=data.remove_usernames, org=org)

        logs = []
        for page in get_pages(data, org):
            new_authors = (set(page.get_authors()) | set(add_users)) - set(remove_users)
            if not new_authors:
                raise BadRequest(f'Page {page.slug} now has no authors. Change remove_usernames')

            if not data.dry_run:
                page.authors.add(*add_users)
                page.authors.remove(*remove_users)

            logs += [f'Page [{page.slug}] added author [{user.username}]' for user in add_users]
            logs += [f'Page [{page.slug}] removed author [{user.username}]' for user in remove_users]

        log_operation(
            user=request.user,
            operation=SupportOperation.CHANGE_AUTHOR,
            request=data.json(),
            details=json.dumps(logs),
        )

        return ChangeAuthorsResponse(changes=logs, dry_run=data.dry_run)
