from typing import List

from ninja import Schema
from pydantic import Field, constr

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.exceptions import BadRequest
from wiki.intranet.models import Staff
from wiki.pages.access import get_bulk_raw_access, interpret_raw_access
from wiki.pages.models import Page
from wiki.pages.models.consts import AclType
from wiki.support_tools.consts import SupportOperation
from wiki.support_tools.models import log_operation
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.org_ctx import org_ctx
from wiki.sync.connect.org_identity import OrganizationIdentity
from wiki.users.models import Group, User
from wiki.utils.supertag import normalize_supertag


class ExplainAccessRequest(Schema):
    slug: str = Field(description='supertag кластера')
    username: constr(min_length=1) = Field(description='username')
    org: OrganizationIdentity = Field(default_factory=OrganizationIdentity, description='Организация')


def get_user(username: str, org: BaseOrganization) -> User:
    try:
        return org.get_users().get(username=username)
    except User.DoesNotExist:
        raise BadRequest(f'Username not found "{username}"')


def get_page(slug: str, org: BaseOrganization) -> Page:
    try:
        return org.get_pages().get(supertag=normalize_supertag(slug))
    except Page.DoesNotExist:
        raise BadRequest(f'Page not found "{slug}"')


class UserSchema(Schema):
    login: str
    name: str
    identity: UserIdentity

    @classmethod
    def serialize(cls, staff: Staff) -> 'UserSchema':
        return UserSchema(
            login=staff.user.username,
            name=staff.get_full_name(),
            identity=UserIdentity.from_user(staff.user),
        )


class GroupSchema(Schema):
    name: str

    @classmethod
    def serialize(cls, group: Group) -> 'GroupSchema':
        return GroupSchema(name=group.name)


class PageAccessSchema(Schema):
    inherits_from: str = None
    type: AclType
    page_authors: List[UserSchema]
    users: List[UserSchema]
    groups: List[GroupSchema]


class ExplainAccessResponse(Schema):
    user_membership: List[GroupSchema]
    page_access: PageAccessSchema


def explain_access(page: Page) -> PageAccessSchema:
    raw_access = get_bulk_raw_access([page])[page]
    access = interpret_raw_access(raw_access)
    accesstype = AclType.DEFAULT

    if access['is_restricted']:
        accesstype = AclType.CUSTOM

    if access['is_owner']:
        accesstype = AclType.ONLY_AUTHORS

    return PageAccessSchema(
        inherits_from=raw_access['latest_supertag']
        if raw_access['list'] and raw_access['latest_supertag'] != raw_access['supertag']
        else None,
        type=accesstype,
        page_authors=[UserSchema.serialize(user.staff) for user in page.authors.all()],
        groups=[GroupSchema.serialize(group) for group in access['groups']],
        users=[UserSchema.serialize(staff) for staff in access['users']],
    )


def explain_access_view(request, data: ExplainAccessRequest) -> ExplainAccessResponse:
    """
    Диагностика почему у пользователя нет доступа.
    Предоставляет группы, в которых состоит пользователь.
    И всех, у кого есть доступ к странице

    Запрос может сделать только пользователь из IDM группы Support
    """

    org = data.org.to_organization()

    with org_ctx(org.as_django_model(), raise_on_empty=False):
        user = get_user(username=data.username, org=org)
        page = get_page(slug=data.slug, org=org)

        response = ExplainAccessResponse(
            user_membership=[GroupSchema.serialize(group) for group in user.get_all_groups()],
            page_access=explain_access(page=page),
        )

        log_operation(
            user=request.user,
            operation=SupportOperation.DIAG_403,
            request=data.json(),
            details=response.json(),
        )

    return response
