from ninja import Schema
from pydantic import Field
from wiki.api_v2.exceptions import BadRequest
from wiki.api_v2.schemas import StatusOk
from wiki.pages.models import Page
from wiki.pages.utils.resurrect import resurrect_page
from wiki.support_tools.consts import SupportOperation
from wiki.support_tools.models import log_operation
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.sync.connect.org_identity import OrganizationIdentity
from wiki.utils.django_redlock.redlock import RedisLock


class RestorePageRequest(Schema):
    slug: str = Field(description='slug кластера')
    org: OrganizationIdentity = Field(default_factory=OrganizationIdentity, description='Организация')


def find_deleted_page(slug: str, org: BaseOrganization) -> Page:
    page = (
        org.get_pages()
        .filter(
            supertag__startswith='.deleted.',
            supertag__endswith='/' + slug,
            status=0,
        )
        .first()
    )
    if page:
        return page
    raise BadRequest(f'Page not found by slug="{slug}"')


def restore_page_view(request, data: RestorePageRequest) -> StatusOk:
    """
    Восстановить страницу по slug
    Запрос может сделать только пользователь из IDM группы Support
    """

    org = data.org.to_organization()

    with RedisLock(name=f'RestorePage: {data.slug}'):
        page = find_deleted_page(slug=data.slug, org=org)
        resurrect_page(page)

        log_operation(
            user=request.user,
            operation=SupportOperation.RESTORE_PAGE,
            request=data.json(),
            details=f'Page [{data.slug}] restored. org_id="{data.org}"',
        )

    return StatusOk()
