import logging
import ssl
import time
from typing import Dict, List

import grpc

from google.protobuf.json_format import MessageToDict
from django.conf import settings

from yandex.cloud.priv.organizationmanager.v1 import (
    organization_service_pb2_grpc,
    organization_service_pb2,
    operation_service_pb2,
    operation_service_pb2_grpc,
)

from yandex.cloud.priv.organizationmanager.v1.transitional import (
    organization_service_pb2_grpc as organization_service_pb2_grpc_transitional,
    organization_service_pb2 as organization_service_pb2_transitional,
)


class GrpcCloudClientError(Exception):
    pass


class OrganizationCreationError(GrpcCloudClientError):
    pass


class OrganizationEnsureError(GrpcCloudClientError):
    pass


class OrganizationNotFoundError(GrpcCloudClientError):
    pass


NEW_ORG_CLOUD_NAME = 'new-organization'


class GrpcCloudClient(object):
    def __init__(self):
        self._creds = None
        self._host = None

    def _get_host(self):
        if not self._host:
            self._host = settings.CLOUD_ORG_HOST

        return self._host

    def _get_creds(self):
        if not self._creds:
            host, port = self._get_host().split(':')
            cert = ssl.get_server_certificate((host, port))
            self._creds = grpc.ssl_channel_credentials(cert.encode('utf-8'))

        return self._creds

    def _get_token(self, user_iam_token: str) -> str:
        if not user_iam_token:
            raise GrpcCloudClientError('IAM token required')

        return f'Bearer {user_iam_token}'

    def get_organization(self, org_id: str, user_iam_token: str) -> Dict[str, str]:
        with grpc.secure_channel(self._get_host(), self._get_creds()) as channel:
            stub = organization_service_pb2_grpc.OrganizationServiceStub(channel)
            try:
                resp = stub.Get(
                    organization_service_pb2.GetOrganizationRequest(organization_id=str(org_id)),
                    metadata=(('authorization', self._get_token(user_iam_token)),),
                )
                return MessageToDict(resp)
            except grpc.RpcError as err:
                status_code = err.code()
                if grpc.StatusCode.NOT_FOUND == status_code:
                    raise OrganizationNotFoundError(f'Organization with id={org_id} not found')
                raise

    def get_operation(self, operation_id: str, user_iam_token: str):
        with grpc.secure_channel(self._get_host(), self._get_creds()) as channel:
            stub = operation_service_pb2_grpc.OperationServiceStub(channel)
            return stub.Get(
                operation_service_pb2.GetOperationRequest(operation_id=operation_id),
                metadata=(('authorization', self._get_token(user_iam_token)),),
            )

    def create_organization(self, org_name: str, user_iam_token: str) -> str:
        data = {
            'name': NEW_ORG_CLOUD_NAME,
            'title': org_name,
        }

        with grpc.secure_channel(self._get_host(), self._get_creds()) as channel:
            stub = organization_service_pb2_grpc.OrganizationServiceStub(channel)

            operation = stub.Create(
                organization_service_pb2.CreateOrganizationRequest(**data),
                metadata=(('authorization', self._get_token(user_iam_token)),),
            )

            while not operation.done:
                operation = self.get_operation(operation.id, user_iam_token)
                time.sleep(1)

            err = MessageToDict(operation.error)

            if err:
                raise OrganizationCreationError(err)

            return MessageToDict(operation.response)['id']

    def ensure_org_in_connect(self, cloud_org_id: str, user_iam_token: str) -> str:
        """Для создания организации в коннекте на основе облачной. При повторном запросе - возвращает уже созданную"""
        data = {
            'organization_id': cloud_org_id,
        }

        with grpc.secure_channel(self._get_host(), self._get_creds()) as channel:
            stub = organization_service_pb2_grpc_transitional.OrganizationServiceStub(channel)

            operation = stub.EnableServices(
                organization_service_pb2_transitional.EnableServicesRequest(**data),
                metadata=(('authorization', self._get_token(user_iam_token)),),
            )

            while not operation.done:
                operation = self.get_operation(operation.id, user_iam_token)
                time.sleep(1)

            err = MessageToDict(operation.error)

            if err:
                raise OrganizationEnsureError(err)

            return MessageToDict(operation.response)['directoryOrganizationId']

    def list_organizations(
        self, user_iam_token: str, page_size=1000, page_token=None, filter=None, cloud_subject=None
    ) -> List[Dict[str, str]]:
        with grpc.secure_channel(self._get_host(), self._get_creds()) as channel:
            stub = organization_service_pb2_grpc.OrganizationServiceStub(channel)
            resp = stub.List(
                organization_service_pb2.ListOrganizationsRequest(
                    page_size=page_size,
                    page_token=page_token,
                    filter=filter,
                    subject_id=cloud_subject,
                ),
                metadata=(('authorization', self._get_token(user_iam_token)),),
            )

            response = MessageToDict(resp)

            if 'organizations' in response:
                return response['organizations']
            else:
                logging.info(f'No user`s cloud organizations: details: ${response}')
                return []
