import logging
import time
from uuid import uuid4

from ylog.context import log_context
from django.core.cache import caches

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.sync.cloud.client import OrganizationCreationError, OrganizationEnsureError
from wiki.sync.connect.tasks.consts import (
    CACHE_FORCED_SYNC_TIMEOUT,
    CLOUD_SYNC_CONNECT_TIMEOUT,
    ForcedSyncError,
    ForcedSyncErrorCode,
    ForcedSyncResult,
    ForcedSyncStatus,
    NEW_ORG_CONNECT_NAME,
)
from wiki.sync.connect.tasks.helpers import (
    ForcedSyncData,
    create_new_org_in_cloud,
    enable_new_org,
    ensure_cloud_org_in_connect,
    get_user_orgs_from_cloud,
    get_user_orgs_from_connect,
    get_user_orgs_from_connect_by_cloud_org_id,
    has_organization_in_wiki_db,
    import_missing_users,
    import_org_data_by_dir_org_id,
    import_user_by_cloud_uid,
    import_user_by_uid,
    is_service_ready,
    noop,
    sync_changes_in_org,
    user_belongs_to_org,
    user_exists_in_local_db,
)
from wiki.utils.tasks.base import LockedCallableTask

logger = logging.getLogger(__name__)

CACHE_FORCED_SYNC_BACKEND = caches['forced_sync']


def get_cache_key(dir_org_id: str, cloud_org_id: str, user_cloud_uid: str, user_uid: str) -> str:
    return f'_{dir_org_id}_{cloud_org_id}_{user_cloud_uid}_{user_uid}'


class ForcedSyncTask(LockedCallableTask):
    name = 'wiki.sync.connect.forced_sync'
    time_limit = 60 * 10  # 600 сек
    max_retries = 3
    lock_name_tpl = 'forced_sync_{dir_org_id}_{cloud_org_id}_{user_cloud_uid}_{user_uid}'

    def run(
        self,
        dir_org_id: str,
        cloud_org_id: str,
        user_cloud_uid: str,
        user_uid: str,
        user_iam_token: str,
        *args,
        **kwargs,
    ):
        with log_context(
            sync_uuid=str(uuid4()),
            user_uid=user_uid,
            user_cloud_uid=user_cloud_uid,
            dir_org_id=dir_org_id,
            cloud_org_id=cloud_org_id,
        ):
            data = ForcedSyncData(
                dir_org_id=dir_org_id,
                cloud_org_id=cloud_org_id,
                user_cloud_uid=user_cloud_uid,
                user_uid=user_uid,
                user_iam_token=user_iam_token,
            )

            cache_key = get_cache_key(data.dir_org_id, data.cloud_org_id, data.user_cloud_uid, data.user_uid)
            result = self.sync(data)
            CACHE_FORCED_SYNC_BACKEND.set(cache_key, result.dict(), CACHE_FORCED_SYNC_TIMEOUT)
            logger.info(f'Storing result {cache_key} - {result.dict()}')

    @classmethod
    def sync(cls, data: ForcedSyncData) -> ForcedSyncResult:
        logger.info('Forced Sync org')

        for method in [
            cls.ensure_at_least_one_org,
            cls.ensure_org_in_local_db,
            cls.ensure_user_in_db,
            cls.ensure_user_belongs_to_org,
            cls.ensure_wiki_enabled_in_org,
        ]:
            try:
                method(data)
                logger.info(f'Sync operation {method.__name__} was successful')
            except ForcedSyncError as err:
                logger.exception(f'Sync operation {method.__name__} was unsuccessful')
                return ForcedSyncResult(result=ForcedSyncStatus.FAILED, error_code=err.code)
            except Exception:
                logger.exception(f'Sync operation {method.__name__} has unhandled exception"')
                return ForcedSyncResult(
                    result=ForcedSyncStatus.FAILED, error_code=ForcedSyncErrorCode.UNHANDLED_EXCEPTION
                )

        return ForcedSyncResult(result=ForcedSyncStatus.OK, org_id=data.dir_org_id)

    @classmethod
    def ensure_user_in_db(cls, data: ForcedSyncData):
        # Обеспечить наличие пользователя в БД.
        identity = UserIdentity(uid=data.user_uid, cloud_uid=data.user_cloud_uid)

        for method in [
            noop,
            import_user_by_uid,
            import_user_by_cloud_uid,
        ]:
            method(data)
            if user_exists_in_local_db(identity):
                return

        # после всех наших попыток ничего не сработало
        raise ForcedSyncError(code=ForcedSyncErrorCode.DIR_HAS_NO_CLOUD_USER)

    @classmethod
    def ensure_at_least_one_org(cls, data: ForcedSyncData):
        # Обеспечить наличие у пользователя хотя бы одной организации
        if data.dir_org_id:
            return

        # Если он облачный (нету x-uid, у нас бо-о-о-ольшие проблемы :) )
        if not data.user_uid:
            raise ForcedSyncError(code=ForcedSyncErrorCode.ORPHAN_FED_USER)

        # Может у него в коннекте есть организации?
        if data.cloud_org_id:
            connect_org = get_user_orgs_from_connect_by_cloud_org_id(
                user_uid=data.user_uid, user_cloud_uid=data.user_cloud_uid, cloud_org_id=data.cloud_org_id
            )
        else:
            connect_orgs = get_user_orgs_from_connect(user_uid=data.user_uid, user_cloud_uid=data.user_cloud_uid)
            connect_org = connect_orgs[0] if connect_orgs else None

        if connect_org:
            data.dir_org_id = connect_org['id']
            return

        # Найдем/создадим облачную организацию для пользователя
        logger.info(f'No dir_org_id provided, will try to create cloud orgs. given cloud_id = ${data.cloud_org_id}')
        cloud_orgs = get_user_orgs_from_cloud(data.user_iam_token)
        cloud_orgs_id = {cloud_org['id'] for cloud_org in cloud_orgs}

        if data.cloud_org_id and data.cloud_org_id not in cloud_orgs_id:
            logger.exception(f'Given cloud_id ${data.cloud_org_id} not found in user`s cloud_org')
            raise ForcedSyncError(code=ForcedSyncErrorCode.NOT_FOUND_CLOUD_ORG_ID)
        elif data.cloud_org_id:
            cloud_org_id = data.cloud_org_id
            logger.info(f'Given cloud_org ${cloud_org_id} exists')
        elif cloud_orgs_id:
            cloud_org_id = sorted(cloud_orgs_id)[0]
            logger.info(f'Choose the first users`s cloud_org ${cloud_org_id}')
        else:
            logger.info('No cloud orgs, will create one')
            try:
                cloud_org_id = create_new_org_in_cloud(NEW_ORG_CONNECT_NAME, data.user_iam_token)
            except OrganizationCreationError:
                logger.exception('Operation of creating an organization failed')
                raise ForcedSyncError(code=ForcedSyncErrorCode.FAILED_TO_CREATE_NEW_ORG)

        # Отправим запрос в облако на создание организации в коннекте
        try:
            ensure_cloud_org_in_connect(cloud_org_id, data.user_iam_token)
        except OrganizationEnsureError:
            logger.exception('Operation of ensure cloud_org in connect failed')
            raise ForcedSyncError(code=ForcedSyncErrorCode.FAILED_TO_CREATE_NEW_ORG)

        # ждем синка организаций из облака в коннект
        deadline = time.monotonic() + CLOUD_SYNC_CONNECT_TIMEOUT
        while time.monotonic() < deadline:
            org = get_user_orgs_from_connect_by_cloud_org_id(
                user_uid=data.user_uid, user_cloud_uid=data.user_cloud_uid, cloud_org_id=cloud_org_id
            )
            logger.info(f'Waiting for creation new org from cloud in connect; details: ${org}')
            if org:
                data.dir_org_id = org['id']
                data.cloud_org_id = org['cloud_org_id']
                return
            time.sleep(3)
        raise ForcedSyncError(code=ForcedSyncErrorCode.DIR_HAS_NO_NEW_ORG)

    @classmethod
    def ensure_org_in_local_db(cls, data):
        for method in [
            noop,
            import_org_data_by_dir_org_id,
        ]:
            method(data)
            if has_organization_in_wiki_db(data.dir_org_id):
                return

        raise ForcedSyncError(code=ForcedSyncErrorCode.DIR_HAS_NO_ORG)

    @classmethod
    def ensure_user_belongs_to_org(cls, data):
        for method in [noop, sync_changes_in_org, import_missing_users]:
            method(data)
            if user_belongs_to_org(data.dir_org_id, data.user_cloud_uid, data.user_uid):
                return

        raise ForcedSyncError(code=ForcedSyncErrorCode.USER_DOESNT_BELONG_TO_ORG)

    @classmethod
    def ensure_wiki_enabled_in_org(cls, data):
        for method in [
            noop,
            enable_new_org,
        ]:
            method(data)
            if is_service_ready(data.dir_org_id):
                return

        raise ForcedSyncError(code=ForcedSyncErrorCode.FAILED_TO_ENABLE_SVC)
