import logging
import time

from django.core.exceptions import ObjectDoesNotExist
from pydantic import BaseModel
from django.conf import settings
from django.contrib.auth import get_user_model

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.sync.connect.dir_client import (
    DirClient,
    DirClientError,
    OrganizationAlreadyDeletedError,
    ServiceStatusAction,
    UserNotFoundError,
)
from wiki.sync.connect.utils import import_dir_organization as exec_import
from wiki.sync.connect.models import Organization
from wiki.sync.cloud.client import GrpcCloudClient
from wiki.sync.connect.tasks.consts import ForcedSyncError, ForcedSyncErrorCode
from wiki.users.dao import get_user_by_identity, UserNotFound
from wiki.utils.lock import execute_with_lock
from wiki.intranet.models import Staff
from wiki.utils.models import queryset_iterator

logger = logging.getLogger(__name__)

dir_client = DirClient()

cloud_client = GrpcCloudClient()


class ForcedSyncData(BaseModel):
    dir_org_id: str
    user_cloud_uid: str
    user_uid: str
    user_iam_token: str

    cloud_org_id: str = None

    def secure_dict(self) -> dict[str, str | None]:
        msg = self.dict()
        msg['user_iam_token'] = msg.get('user_iam_token', '')[:5]
        return msg


def sync_changes_in_org(data: ForcedSyncData):
    # Проверяем наличие необработанных событий об изменениях в структуре организации и накатываем изменения при
    # необходимости
    from wiki.sync.connect.tasks import ProcessChangeEventsTask

    ProcessChangeEventsTask().run(data.dir_org_id)


def import_user_by_uid(data: ForcedSyncData):
    if data.user_uid:
        try:
            from wiki.users_biz.management.commands.import_missing_users import import_user_directly

            import_user_directly(data.dir_org_id, data.user_uid)
        except UserNotFoundError:
            logger.exception(
                f'Can\'t import user with cloud_id={data.user_cloud_uid} in organization with id={data.dir_org_id}'
            )
        except Exception:
            logger.exception(
                f'Unhandled exception happened id={data.user_uid} in organization with id={data.dir_org_id}'
            )
            raise


def import_user_by_cloud_uid(data: ForcedSyncData):
    if data.user_cloud_uid:
        try:
            from wiki.users_biz.management.commands.import_missing_users import import_user_directly_by_cloud_uid

            import_user_directly_by_cloud_uid(data.dir_org_id, data.user_cloud_uid)
        except UserNotFoundError:
            logger.exception(
                f'Can\'t import user with cloud_id={data.user_cloud_uid} in organization with id={data.dir_org_id}'
            )
            pass
        except Exception:
            logger.exception(
                f'Unhandled exception happened id={data.user_uid} in organization with id={data.dir_org_id}'
            )
            raise


def noop(data: ForcedSyncData):
    pass


def import_missing_users(data: ForcedSyncData):
    try:
        from wiki.users_biz.management.commands.import_missing_users import import_missing_users_directly

        import_missing_users_directly(data.dir_org_id)
    except Exception:
        logger.exception(
            f'Can\'t import user with uid={data.user_uid} and cloud_uid={data.user_cloud_uid} '
            f'in organization with id={data.dir_org_id}'
        )
        raise


def import_dir_organization_mp_safe(org, dir_org_id):
    # может быть ее кто-то уже импортирует в этот момент
    execute_with_lock(
        'dir_sync_import_org_%s' % dir_org_id, lambda: _import_dir_organization_unsafe(org, dir_org_id), timeout=30
    )


def _import_dir_organization_unsafe(org, dir_org_id):
    if has_organization_in_wiki_db(dir_org_id):
        return  # к моменту, как мы получили лок, организацию уже кто-то создал

    organization = exec_import(org)
    if not organization:
        logger.error('Could not import organization %s', dir_org_id)


def has_wiki_bot(dir_org_id):
    users, _ = dir_client.get_users(dir_org_id)

    for user in users:
        if user.get('service_slug') == 'wiki' and user.get('is_robot'):
            return True
    return False


def has_organization_in_wiki_db(dir_org_id: str) -> bool:
    return Organization.objects.filter(dir_id=dir_org_id).exists()


def get_organization_by_dir_id(dir_org_id: str) -> Organization:
    return Organization.objects.get(dir_id=dir_org_id)


def user_belongs_to_org(dir_org_id: str, user_cloud_uid: str, user_uid: str) -> bool:
    if user_cloud_uid and get_user_model().objects.filter(orgs__dir_id=dir_org_id, cloud_uid=user_cloud_uid).exists():
        return True
    if user_uid and Staff.objects.filter(user__orgs__dir_id=dir_org_id, uid=user_uid).exists():
        return True
    return False


def import_org_data_by_dir_org_id(data: ForcedSyncData):
    try:
        dir_org = dir_client.get_organization(data.dir_org_id, fields=settings.DIRSYNC_FIELDS_ORGANIZATION_FOR_SYNC)
        if dir_org.get('organization_type') == 'portal':
            raise ForcedSyncError(code=ForcedSyncErrorCode.DIR_THIS_IS_PORTAl)

        import_dir_organization_mp_safe(dir_org, data.dir_org_id)
    except DirClientError as err:
        logger.error(f'Can\'t find organization with dir_id={data.dir_org_id} in Connect', repr(err))
        raise ForcedSyncError(code=ForcedSyncErrorCode.DIR_HAS_NO_ORG)
    except Exception:
        logger.exception(f'Can\'t import organization with dir_org_id={data.dir_org_id}')
        raise


def create_new_org_in_connect(org_name: str, user_uid: str) -> str:
    return dir_client.create_organization(org_name, user_uid)


def user_exists_in_local_db(identity: UserIdentity):
    try:
        get_user_by_identity(identity, limit_users_by_current_org=False)
    except UserNotFound:
        return False

    return True


def create_new_org_in_cloud(org_name: str, user_iam_token: str) -> str:
    cloud_org_id = cloud_client.create_organization(org_name, user_iam_token=user_iam_token)
    # NOTE: облако не создает организацию в коннекте, если нужно, то после вызвать ensure_cloud_org_in_connect
    return cloud_org_id


def ensure_cloud_org_in_connect(cloud_org_id: str, user_iam_token: str) -> str:
    dir_id = cloud_client.ensure_org_in_connect(cloud_org_id, user_iam_token=user_iam_token)
    return dir_id


def enable_new_org(data: ForcedSyncData):
    dir_org = dir_client.get_organization(data.dir_org_id, fields=settings.DIRSYNC_FIELDS_ORGANIZATION_FOR_SYNC)
    admin_uid = dir_org.get('admin_id')

    dir_client.change_service_status(data.dir_org_id, ServiceStatusAction.ENABLE, admin_uid or data.user_uid)
    dir_client.post_service_ready('wiki', data.dir_org_id)

    org = get_organization_by_dir_id(data.dir_org_id)
    org.status = Organization.ORG_STATUSES.enabled
    org.save()

    # --- проактивно добавим робота Вики ---

    try:
        wiki_robot = get_user_model().objects.get(dir_id=settings.ROBOT_DIR_ID)
        org = get_organization_by_dir_id(data.dir_org_id)
        wiki_robot.orgs.add(org)
    except ObjectDoesNotExist:
        logger.error('No wiki robot presented!')


def is_service_ready(dir_org_id: str) -> bool:
    org = get_organization_by_dir_id(dir_org_id)
    enabled, ready = dir_client.get_service_status(dir_org_id)
    return enabled and ready and org.status == Organization.ORG_STATUSES.enabled


def get_user_orgs_from_cloud(user_iam_token: str) -> list[dict[str, str]]:
    return cloud_client.list_organizations(user_iam_token)


def get_user_orgs_from_connect(user_uid: str = None, user_cloud_uid: str = None) -> list[dict[str, ...]]:
    return dir_client.get_organizations(user_uid=user_uid, user_cloud_uid=user_cloud_uid)


def get_org_from_connect(dir_org_id: str) -> dict[str, any]:
    return dir_client.get_organization(dir_org_id)


def get_user_orgs_from_connect_by_cloud_org_id(
    user_uid: str = None, user_cloud_uid: str = None, cloud_org_id: str = None
) -> dict[str, ...] | None:
    orgs = get_user_orgs_from_connect(user_uid=user_uid, user_cloud_uid=user_cloud_uid)
    for org in orgs:
        if org['cloud_org_id'] == cloud_org_id:
            return org

    return None


def update_local_cloud_id(rate: int = 10):
    t = 1 / rate
    for org in queryset_iterator(Organization.objects.filter(cloud_id=None), chunk_size=100):
        time.sleep(t)
        try:
            cloud_id = dir_client.get_organization(org.dir_id)['cloud_org_id']
            logger.info(f'{org.id} -> {cloud_id}')
            org.cloud_id = cloud_id
            org.save()
        except OrganizationAlreadyDeletedError:
            logger.info(f'organization {org.id} is deleted')
