import logging
from itertools import groupby

from django.db import transaction
from django.utils import timezone
from ylog.context import log_context
from celery.task import task

from wiki.sync.connect import signals
from wiki.sync.connect.logic import (
    OrganizationSyncStatistics,
    context_from_dir_event,
    disable_organization,
    make_sync_random_for_req_id,
)
from wiki.sync.connect.models import Organization, ChangeEvent
from wiki.sync.connect.dir_client import (
    DirClient,
    DirClientError,
    OrganizationAlreadyDeletedError,
    ServiceIsNotEnabledError,
    OrganizationNotFoundError,
)
from wiki.sync.connect.tasks.helpers import has_wiki_bot, import_dir_organization_mp_safe
from wiki.utils.tasks.base import LockedCallableTask
from wiki.utils import lock
from wiki.sync.connect.utils import ensure_service_ready as _ensure_service_ready


logger = logging.getLogger(__name__)


class ProcessChangeEventsTask(LockedCallableTask):
    """
    Проверить наличие событий об изменениях в структуре организации в Директории с необработанными номерами ревизий
    и при наличии таковых импортировать изменения.
    Задача запускается для организации с переданным dir_org_id.
    """

    name = 'wiki.sync.connect.sync_org_changes'
    logger = logging.getLogger(name)
    time_limit = 60 * 30  # 1800 сек
    lock_name_tpl = 'sync_dir_data_changes_{dir_org_id}'

    def run(self, dir_org_id, req_id_from_dir=None, *args, **kwargs):
        dir_client = DirClient(sync_random=make_sync_random_for_req_id(), req_id_from_dir=req_id_from_dir)

        with log_context(celery_task_name='sync_dir_data_changes_for_org', req_id=dir_client.get_req_id()):
            logger.info('Sync-2 changes of data from Directory for organization: %s', dir_org_id)

            org_status = Organization.objects.filter(dir_id=dir_org_id).values_list('status', flat=True).first()
            if org_status and org_status == Organization.ORG_STATUSES.disabled:
                logger.info('Organization with dir_id={} is disabled.'.format(dir_org_id))
                return

            try:
                org = dir_client.get_organization(dir_org_id)

                # проверим, что организация не заблокирована в Директории
                if org['is_blocked']:
                    # блокируем организацию и у нас и обновления не обрабатываем.
                    logger.info('Organization with dir_id={} is blocked in Directory.'.format(dir_org_id))
                    disable_organization(dir_org_id)
                    return

                # проверим, что организация существует в базе Вики
                if not Organization.objects.filter(dir_id=dir_org_id).exists():
                    if not has_wiki_bot(dir_org_id):
                        logger.info('Organization with id=%s has no wiki bot, we gonna get it next time', dir_org_id)
                        return
                    import_dir_organization_mp_safe(org, dir_org_id)

            except (OrganizationAlreadyDeletedError, ServiceIsNotEnabledError):
                disable_organization(dir_org_id)
                return
            except DirClientError as err:
                logger.warning('Can\'t check status of organization with dir_id=%s: %s', dir_org_id, repr(err))
                return

            revision = ChangeEvent.objects.get(org__dir_id=dir_org_id).revision

            with log_context(
                dir_org_id=str(dir_org_id), last_revision=str(revision), req_id=dir_client.get_req_id(str(dir_org_id))
            ):
                try:
                    params = {'per_page': 50}
                    if revision:
                        params['revision__gt'] = revision
                    events = dir_client.get_events(dir_org_id, **params)
                except OrganizationNotFoundError:
                    return
                except DirClientError as err:
                    content = err.response.json()
                    if err.response.status_code == 403 and (
                        content.get('code') == 'service_was_disabled' or content.get('code') == 'service_is_not_enabled'
                    ):
                        # у организации в Директории отключен наш сервис. Отключим организацию в нашей базе,
                        # если она не отключена
                        disable_organization(dir_org_id)
                    else:
                        # TODO: синхронизация организации сломана, отражать это в статистике
                        logger.exception(
                            'Error while retrieving events for organization dir_id=%s: "%s"', dir_org_id, repr(err)
                        )
                    return
                except Exception as err:
                    # TODO: синхронизация организации сломана, отражать это в статистике
                    logger.exception(
                        'Error while retrieving events for organization dir_id=%s: "%s"', dir_org_id, repr(err)
                    )
                    return

                if events:
                    try:
                        with OrganizationSyncStatistics(dir_org_id):
                            for revision_num, events_iter in groupby(events, key=lambda e: e['revision']):
                                process_change_event(revision_num, list(events_iter), dir_org_id)
                    except Exception as err:
                        logger.exception(
                            'Error processing of change events of organization %s: "%s"', dir_org_id, repr(err)
                        )


ignored_signals = {
    'service_ready',
    'user_alias_added',
    'domain_occupied',
    'domain_added',
    'domain_deleted',
    'organization_owner_changed',
    'service_license_changed',
    'resource_modified',
    'service_robot_added',
    'resource_added',
    'user_alias_deleted',
}


@task(name='dir_data_sync.sync_dir_data_changes')
@lock.get_lock_or_do_nothing('sync_dir_data_changes')
def sync_dir_data_changes(dir_org_ids_list=None, req_id_from_dir=None):
    """
    Проверить наличие событий об изменениях в структуре организации в Директории с необработанными номерами ревизий
    и при наличии таковых импортировать изменения.
    Задача запускается для всех организаций.
    """
    logger.info('Sync changes of data from Directory for organizations: %s', (dir_org_ids_list or 'all'))

    if dir_org_ids_list is None:
        dir_org_ids_list = Organization.objects.exclude(status=Organization.ORG_STATUSES.disabled).values_list(
            'dir_id', flat=True
        )
    else:
        dir_client = DirClient(sync_random=make_sync_random_for_req_id(), req_id_from_dir=req_id_from_dir)

        try:
            dir_org_ids_list = [
                dir_org_id
                for dir_org_id in dir_org_ids_list
                if not dir_client.get_organization(dir_org_id)['is_blocked']
            ]
        except DirClientError as err:
            logger.warning('Can\'t check status of organization with dir_id=%s', repr(err))
            return

    for dir_org_id in dir_org_ids_list:
        ProcessChangeEventsTask().delay(dir_org_id=dir_org_id, req_id_from_dir=req_id_from_dir)


@transaction.atomic
def process_change_event(revision_num, events, dir_org_id):
    last_change_event = ChangeEvent.objects.select_for_update().get(org__dir_id=dir_org_id)
    if last_change_event.revision and last_change_event.revision >= revision_num:
        # получили событие которое уже обрабатывали.
        return
    try:
        for event in events:
            signal_name = event['name'].lower()
            if signal_name in ignored_signals:
                continue
            with log_context(celery_task_name='sync_dir_data_changes', **context_from_dir_event(event)):
                logger.info('Applying event %s: org %s, rev %s' % (event['name'], dir_org_id, revision_num))

                try:
                    signal = getattr(signals, signal_name)
                except AttributeError:
                    msg = 'There is no signal for event [%s]'
                    logger.warn(msg, event['name'])
                else:
                    # к нам сейчас события прилетают без атрибута org_id, поэтому чтобы вписаться в наш общий формат
                    # обработки событий - добавим в объект этот атрибут
                    if signal_name in ('service_enabled', 'service_disabled'):
                        event['object']['org_id'] = dir_org_id

                    # https://st.yandex-team.ru/DIR-4513
                    # У события с типом group_membership_changed может отсутствовать секция object,
                    # поэтому сейчас чтобы не падать при обработке таких событий добавим object с id орг-ции
                    if signal_name == 'group_membership_changed' and 'object' not in event:
                        event['object'] = {'org_id': dir_org_id, 'id': event['id']}

                    signal.send(
                        sender=sync_dir_data_changes,  # где-то в логике обработки сигналов захардкожено имя функции (!)
                        object=event['object'],
                        content=event['content'],
                    )
    except Exception:
        raise
    else:
        last_change_event.revision = revision_num
        last_change_event.last_pull_at = timezone.now()
        last_change_event.save()


@task(name='dir_data_sync.ensure_service_ready')
def ensure_service_ready():
    _ensure_service_ready()


# синоним чтобы перехватить обработку всех задач из dir_data_sync
class ProcessChangeEventsTaskAlias(ProcessChangeEventsTask):
    name = 'wiki.sync_dir_data_changes_for_org'
    lock_name_tpl = 'sync_dir_data_changes_{dir_org_id}'
    logger = logging.getLogger(name)
