import logging

from wiki.sync.connect import signals  # noqa
from wiki.sync.connect.logic import make_sync_random_for_req_id
from wiki.sync.connect.dir_client import DirClient, DirClientError
from wiki.sync.connect.models import Organization
from wiki.sync.connect.tasks.process_changes import ProcessChangeEventsTask
from wiki.utils.tasks.base import LockedCallableTask

logger = logging.getLogger(__name__)


class SyncConnectTask(LockedCallableTask):
    name = 'wiki.sync.connect.sync_changes'
    logger = logging.getLogger(name)
    time_limit = 60 * 300  # 18 000 сек
    lock_name_tpl = 'sync_connect'

    def run(self, dir_org_ids_list=None, req_id_from_dir=None, *args, **kwargs):
        """
        Проверить наличие событий об изменениях в структуре организации в Директории с необработанными номерами ревизий
        и при наличии таковых импортировать изменения.
        Задача запускается для всех организаций.
        """
        logger.info('Sync-2 changes of data from Directory for organizations: %s', (dir_org_ids_list or 'all'))

        if dir_org_ids_list is None:
            dir_org_ids_list = Organization.objects.exclude(status=Organization.ORG_STATUSES.disabled).values_list(
                'dir_id', flat=True
            )
        else:
            dir_client = DirClient(sync_random=make_sync_random_for_req_id(), req_id_from_dir=req_id_from_dir)

            try:
                dir_org_ids_list = [
                    dir_org_id
                    for dir_org_id in dir_org_ids_list
                    if not dir_client.get_organization(dir_org_id)['is_blocked']
                ]
            except DirClientError as err:
                logger.warning('Can\'t check status of organization with dir_id=%s', repr(err))
                return

        for dir_org_id in dir_org_ids_list:
            ProcessChangeEventsTask().delay(dir_org_id=dir_org_id, req_id_from_dir=req_id_from_dir)
