import logging

from celery.task import task

from ylog.context import log_context

from wiki.sync.connect.logic import make_sync_random_for_req_id
from wiki.sync.connect.models import Organization
from wiki.sync.connect.dir_client import DirClient
from wiki.sync.connect.tasks.helpers import has_wiki_bot, import_dir_organization_mp_safe
from wiki.utils.tasks.base import LockedCallableTask

logger = logging.getLogger(__name__)


@task(name='wiki.sync.create_new_organization', ignore_result=True)
def create_new_organization(dir_org_obj):
    """
    Задача создания новой организации на основе данных организации из Директории.
    """
    dir_org_id = dir_org_obj['id']
    with log_context(dir_org_id=dir_org_id):
        logger.info('Importing new organization with id=%s', dir_org_id)

        # С момента включения сервиса до момента добавления бота
        # может пройти какое-то время. Не стоит добавлять организацию
        # без бота, так как мы упадем на этапе создания страницы.

        if not has_wiki_bot(dir_org_id):
            logger.info('Organization with id=%s has no wiki bot, we gonna get it next time', dir_org_id)
            return

        import_dir_organization_mp_safe(dir_org_obj, dir_org_obj['id'])


class SyncNewOrganizationsTask(LockedCallableTask):
    """
    Найти организации в Директории, которые уже подключили у себя сервис, но про которые сам сервис еще не знает.
    Поставить для каждой такой организации отложенную задачу по импорту ее данных в базу сервиса.
    """

    name = 'wiki.sync.connect.sync_new_organizations'
    logger = logging.getLogger(name)
    time_limit = 60 * 10
    lock_name_tpl = 'sync_new_organizations'

    def run(*args, **kwargs):
        """
        Найти организации в Директории, которые уже подключили у себя сервис, но про которые сам сервис еще не знает.
        Поставить для каждой такой организации отложенную задачу по импорту ее данных в базу сервиса.
        """
        dir_client = DirClient(sync_random=make_sync_random_for_req_id())

        with log_context(celery_task_name='sync_new_organizations', req_id=dir_client.get_req_id()):
            dir_org_ids = Organization.objects.values_list('dir_id', flat=True)
            org_list_from_dir = dir_client.get_organizations(**{'service.ready': False})
            org_map_from_dir = {str(org['id']): org for org in org_list_from_dir if not org['is_blocked']}

            not_existed_ids = set(org_map_from_dir.keys()) - set(dir_org_ids)

            for dir_org_id in not_existed_ids:
                dir_org_obj = org_map_from_dir[dir_org_id]
                create_new_organization.delay(dir_org_obj)
