
from wiki.sync.staff.mapping.exceptions import RemoteObjectNotFound
from wiki.utils.staff import get_staff_repository


def fmt_as_iso(dt):
    return dt.strftime('%Y-%m-%dT%H:%M:%S.%f%z')


class DataSource(object):
    """
    Источник данных.

    Репозиторий сервиса staff с параметрами какие поля из него забирать и доп фильтрами.
    """

    def __init__(self, repository_name, extra_lookups, fields):

        self.repository_name = repository_name
        self.extra_lookups = extra_lookups
        # поля id и _meta.modified_at используются в сортировках и update, не будем указывать отдельно

        self.fields = ','.join(set(fields).union({'id', '_meta.modified_at'}))

    def get_repository(self):
        return get_staff_repository(self.repository_name)

    def get_query(self, id_from=None, sync_from=None, sync_to=None, sort=None, limit=500):

        lookups = self.extra_lookups.copy()

        lookups.update(
            {
                '_limit': limit,
                '_fields': self.fields,
            }
        )

        if sort:
            lookups['_sort'] = sort

        query = []
        if id_from:
            query.append('id>%s' % id_from)
        if sync_from:
            query.append('_meta.modified_at >= "%s"' % (fmt_as_iso(sync_from)))
        if sync_to:
            query.append('_meta.modified_at < "%s"' % (fmt_as_iso(sync_to)))

        if len(query) > 0:
            lookups['_query'] = ' and '.join(query)

        return lookups

    def get_objects(self, sync_from=None, sync_to=None, sort=None, limit=500):
        repo = self.get_repository()

        pages = repo.getiter(
            lookup=self.get_query(sync_from=sync_from, sync_to=sync_to, sort=sort), limit=limit
        ).get_pages()

        for page in pages:
            yield page.result

    def get_object(self, lookup_dict):
        repo = self.get_repository()
        lookup = self.get_query(limit=1)

        lookup.update(lookup_dict)

        objects = repo.getiter(lookup=lookup).first_page.result
        if len(objects) == 0:
            raise RemoteObjectNotFound(
                'Repository "%s" dont have any object. Lookup was %s' % (self.repository_name, lookup)
            )
        return objects[-1]
