
from typing import Type  # no qa

from wiki.sync.staff.logger import sync_logger
from wiki.sync.staff.mapping.models import mapper_registry
from wiki.sync.staff.mapping.repository_mapper import RepositoryMapper  # no qa
from wiki.sync.staff.utils import datetime_from_str


def do_update(datasource, mappers, from_date, to_date=None, batch_size=500):
    """
    Получить из источника данных объекты начиная с даты from_date и применить обновления через мапперы.

    Объекты забираются в порядке возразтания даты обновления от самой старой к самой новой.
    При этом процесс закончится когда нельзя получить новую пачку.

    Так как "от даты" включает в себя нижнюю границу (так как может быть много измненений в одну секудну)
    нам важно трекать что мы уже успели применить и не обновилась ли модель еще раз.
    """
    processed_ids = {}

    total_failures_count = 0
    total_success_count = 0

    for raw_batch in datasource.get_objects(
        sync_from=from_date, sync_to=to_date, limit=batch_size, sort='_meta.modified_at'
    ):

        batch = []

        if len(raw_batch) == 0:
            return

        for remote_raw in raw_batch:
            try:
                pk = remote_raw['id']
                mod_dt = remote_raw['_meta']['modified_at']
            except KeyError:
                sync_logger.error('Got malformed update model. %s' % remote_raw)
                continue

            if pk in processed_ids and processed_ids[pk] == mod_dt:
                continue

            processed_ids[pk] = mod_dt
            batch.append(remote_raw)

        if len(batch) == 0:
            continue

        for mapper in mappers:  # type: Type[RepositoryMapper]
            success_count, failures_count = mapper.process_batch(batch, mapper_registry=mapper_registry)
            total_failures_count += failures_count
            total_success_count += success_count

        from_date = datetime_from_str(batch[-1]['_meta']['modified_at'])
        yield from_date, total_success_count, total_failures_count
