from wiki.intranet.models import Staff
from wiki.intranet.models.intranet_extensions import Group, GroupMembership
from wiki.sync.staff.logger import sync_logger
from wiki.utils.staff import get_staff_groupmembership_repo


def do_group_resync(group, dry_run=True, batch_size=500):
    # type: (Group, bool, int) -> (int,int)
    g_ids = {g.id for g in group.get_descendants()}
    g_ids.add(group.id)

    outdated_groupmembership_dict, missing_groupmembership = get_sync_status()
    sync_logger.info('%s GroupMembership are marked for deletion' % len(outdated_groupmembership_dict))
    sync_logger.info('%s GroupMembership are to be inserted' % len(missing_groupmembership))
    outdated_groupmembership_f, missing_groupmembership_f = filterout_data_by_group_ids(
        g_ids, outdated_groupmembership_dict, missing_groupmembership
    )
    if dry_run:
        return 0, 0
    success_rm, failure_rm = 0, 0
    # outdated_groupmembership_pks = list(outdated_groupmembership_dict.values())

    # success_rm, failure_rm = _batch_gm_cleanup(outdated_groupmembership_pks, batch_size)
    success_create, failure_create = _gm_create(missing_groupmembership_f)

    return success_rm + success_create, failure_rm + failure_create


def filterout_data_by_group_ids(g_ids, outdated_groupmembership_dict, missing_groupmembership):
    # outdated_groupmembership_dict : (staff__uid, group_id) -> gm.pk
    # missing_groupmembership : [(staff_uid, group_id)]
    missing_groupmembership_f = [m for m in missing_groupmembership if m[1] in g_ids]
    outdated_groupmembership_dict_f = {k: v for k, v in list(outdated_groupmembership_dict.items()) if k[1] in g_ids}
    return outdated_groupmembership_dict_f, missing_groupmembership_f


def do_sync_groupmembership(dry_run=True, batch_size=500):
    outdated_groupmembership_dict, missing_groupmembership = get_sync_status()
    sync_logger.info('%s GroupMembership are to be deleted' % len(outdated_groupmembership_dict))
    sync_logger.info('%s GroupMembership are to be inserted' % len(missing_groupmembership))

    if dry_run:
        return 0, 0

    success_rm, failure_rm = 0, 0
    outdated_groupmembership_pks = list(outdated_groupmembership_dict.values())

    success_rm, failure_rm = _batch_gm_cleanup(outdated_groupmembership_pks, batch_size)
    success_create, failure_create = _gm_create(missing_groupmembership)

    return success_rm + success_create, failure_rm + failure_create


def get_sync_status(sortby='group.id'):
    """
    @return: Tuple[Dict[Tuple(str, int), int], Set[Tuple(str, int)]] - Список id устаревших GroupMembership,
    Сет таплов  staff__uid, group_id
    """
    gm = get_staff_groupmembership_repo()
    local_membership = {
        (k, v): _id for k, v, _id in GroupMembership.objects.values_list('staff__uid', 'group_id', 'id')
    }
    missing_groupmembership = set()

    # local_membership сейчас содержит пары ("staff__uid", "group_id") => GroupMembership.pk
    # проходим отсеивание через все актуальные принадлежности сотрудников

    # пытаемся удалить из словаря те что есть - там останутся только лишние
    # если удалить не удалось - заносим пару в missing

    # потом батчем постараемся удалить лишние по id
    # и создать тех которых нет, учитывая что импорт может отстать а bulk_create должен пройти

    pages_result = gm.getiter({'_fields': 'person.uid,group.id', '_sort': sortby, '_limit': '10000'})

    for page in pages_result.get_pages():
        for record in page.result:
            if not record['person']:  # в тестинге иногда проскакивает person {}
                continue

            staff_uid = record['person']['uid']
            group_id = int(record['group']['id'])

            try:
                del local_membership[(staff_uid, group_id)]
            except KeyError:
                missing_groupmembership.add((staff_uid, group_id))

    return local_membership, missing_groupmembership


def _batch_gm_cleanup(deleted_groupmembership_pks, chunk_size=1000):
    deleted = 0

    for chunk_from in range(0, len(deleted_groupmembership_pks), chunk_size):
        rm_count, _ = GroupMembership.objects.filter(
            id__in=deleted_groupmembership_pks[chunk_from : chunk_from + chunk_size]
        ).delete()
        deleted += rm_count

    return deleted, len(deleted_groupmembership_pks) - deleted


def _gm_create(missing_groupmembership):
    staff_uids = {u[0] for u in missing_groupmembership}
    staff_uid_to_id_map = {uid: _id for uid, _id in Staff.objects.filter(uid__in=staff_uids).values_list('uid', 'id')}
    success = 0
    error = 0
    for staff_uid, group_id in missing_groupmembership:
        try:
            g = GroupMembership(staff_id=staff_uid_to_id_map[staff_uid], group_id=group_id)
            g.save()
            success += 1
            sync_logger.info(f'Creating GM: {staff_uid} x {group_id}')
        except Exception as e:
            error += 1
            sync_logger.warning(f'Creating GM: {staff_uid} x {group_id} failed: {e}')

    return success, error


def get_person_gm_status(staff):
    """
    @return: Tuple[Dict[Tuple(str, int), int], Set[Tuple(str, int)]] - Список id устаревших GroupMembership,
    Сет таплов  staff__uid, group_id
    """
    gm = get_staff_groupmembership_repo()
    local_membership = {
        (k, v): _id
        for k, v, _id in GroupMembership.objects.filter(staff=staff).values_list('staff__uid', 'group_id', 'id')
    }

    missing_groupmembership = set()
    pages_result = gm.getiter({'_fields': 'person.uid,group.id', 'person.id': staff.id, '_limit': '10000'})

    for page in pages_result.get_pages():
        for record in page.result:
            staff_uid = record['person']['uid']
            group_id = int(record['group']['id'])

            try:
                del local_membership[(staff_uid, group_id)]
            except KeyError:
                missing_groupmembership.add((staff_uid, group_id))

    return local_membership, missing_groupmembership
