import datetime

from django.db.models import Min, Value
from django.db.models.functions import Coalesce
import pytz
import waffle

from wiki.api_core.waffle_switches import ENABLE_STAFF_SYNC_V3
from wiki.sync.staff.logger import sync_logger
from wiki.sync.staff.mapping.staff_sync import do_update
from wiki.sync.staff.mapping.sync_groupmembership import do_sync_groupmembership
from wiki.sync.staff.mapping.wiki_fullsync import get_fullsync_projection
from wiki.sync.staff.models import LastUpdate
from wiki.utils import timezone
from wiki.utils.tasks.base import LockedCallableTask

sync_gm_name = 'do_sync_gm'


def sync_staff_membership():
    (last_update, _) = LastUpdate.objects.get_or_create(update_name=sync_gm_name)

    last_update.started_at = timezone.now()
    last_update.save()

    success, failures = do_sync_groupmembership(dry_run=False)

    last_update.updated_at = timezone.now()
    last_update.success_count = success
    last_update.failed_count = failures
    last_update.save()


def sync_full_staff(run_only=None):
    for update_name, datasource, mappers in get_fullsync_projection():
        if run_only is not None and update_name not in run_only:
            sync_logger.info('Skipping %s' % update_name)
            continue

        (last_update, _) = LastUpdate.objects.get_or_create(update_name=update_name)
        last_update.started_at = timezone.now()
        last_update.save()

        total_failed_count = 0
        total_success_count = 0
        last_sync_date = last_update.updated_at

        for syncdate, success_count, failed_count in do_update(
            datasource=datasource, mappers=mappers, from_date=last_sync_date
        ):
            total_success_count += success_count
            total_failed_count += failed_count

            last_update.updated_at = syncdate
            last_update.success_count = total_success_count
            last_update.failed_count = total_failed_count
            last_update.save()

        msg = "Staff -> Local '%s' sync. Success/Failure: %s/%s. Update date advanced from %s to %s." % (
            update_name,
            total_success_count,
            total_failed_count,
            last_sync_date,
            last_update.updated_at,
        )

        if total_failed_count > 0:
            sync_logger.warn(msg)
        elif total_success_count > 0:
            sync_logger.info(msg)


def get_staff_sync_delay():
    update_names = [update_name for update_name, datasource, mappers in get_fullsync_projection()]
    last_update = LastUpdate.objects.filter(update_name__in=update_names).aggregate(
        started_at_min=Coalesce(Min('started_at'), Value(datetime.datetime(2000, 1, 1)))
    )['started_at_min']
    utc_now = datetime.datetime.now(pytz.utc)
    return (utc_now - last_update).total_seconds()


class SyncStaffMembershipTask(LockedCallableTask):
    name = 'wiki.sync_staff_membership'
    logger = sync_logger
    time_limit = 60 * 10

    def run(self, *args, **kwargs):
        if not waffle.switch_is_active(ENABLE_STAFF_SYNC_V3):
            return
        sync_staff_membership()


class SyncStaffModelsTask(LockedCallableTask):
    name = 'wiki.sync_staff_models'
    time_limit = 60 * 60 * 3
    logger = sync_logger

    def run(self, *args, **kwargs):
        if not waffle.switch_is_active(ENABLE_STAFF_SYNC_V3):
            return
        sync_full_staff()
