import pytz
from datetime import datetime
from typing import List

from django.conf import settings
from django.core.cache import caches

from wiki.ping.models import TouchMe
from wiki.unistat.metrics import WFAAS_RPS_METRIC
from wiki.utils.instance_properties import get_environment
from wiki.utils.mongo import get_mongo_client
from wiki.unistat.uwsgi_stat import uwsgi_unistat


CACHE_UNISTAT_METRICS = caches['unistat_metrics']
OAUTH_TOKEN_TTL_CACHE_TIMEOUT = 60 * 60 * 24  # in sec
OAUTH_TOKEN_TTL_CACHE_KEY = f'oauth_token_ttl_{settings.WIKI_CODE}_{get_environment()}_key'


def get_celery_ping_delay():
    try:
        touch_object = TouchMe.objects.get()
    except TouchMe.DoesNotExist:
        return 100500

    celery_last_touch = touch_object.modified_at
    utc_now = datetime.now(pytz.utc)
    return (utc_now - celery_last_touch).total_seconds()


def get_staff_delay():
    if not settings.IS_BUSINESS:
        from wiki.sync.staff.tasks.sync_staff import get_staff_sync_delay  # no qa

        return get_staff_sync_delay()
    return 0


def build_unistat():
    results = []

    results += [['celery_queue_axxx', get_celery_ping_delay()], ['staff_sync_axxx', get_staff_delay()]]
    results += WFAAS_RPS_METRIC.produce_unistat_metric()

    results += get_oauth_token_ttl_data_from_cache()

    if settings.IS_BUSINESS:
        results += get_olap_from_cache()

    if settings.UWSGI_UNISTAT_REPORTING:
        results += uwsgi_unistat(settings.UWSGI_UNISTAT_SOCKET)

    return results


def get_olap_from_cache():
    db = get_mongo_client()
    records = list(db.unistat.find({}, {'_id': False}))
    if len(records):
        return records[0]['data']
    else:
        return []


def store_to_cache(data):
    db = get_mongo_client()
    db.unistat.remove({})
    db.unistat.insert({'data': data})


def get_oauth_token_ttl_data_from_cache() -> List[List]:
    result = []
    data = CACHE_UNISTAT_METRICS.get(OAUTH_TOKEN_TTL_CACHE_KEY)
    if data:
        for name, value in data.items():
            # добавляем в результат отрицательное значение для сигнала,
            # так как Мониторадо пока не поддерживает нисходящие пороги
            result.append([f'ttl_{name}_axxx', -int(value)])

    return result


def store_oauth_token_ttl_data_to_cache(data):
    CACHE_UNISTAT_METRICS.set(OAUTH_TOKEN_TTL_CACHE_KEY, data, OAUTH_TOKEN_TTL_CACHE_TIMEOUT)
