import logging
from django.core.cache import caches

logger = logging.getLogger(__name__)
CACHE_UNISTAT_METRICS = caches['unistat_metrics']


class BaseRedisMetric:
    def __init__(self, name: str, suffix: str = 'ammm'):
        self.name = name
        self.suffix = suffix

    def get_unistat_name(self):
        return f'{self.name}_{self.suffix}'

    def produce_unistat_metric(self):
        val = [[self.get_unistat_name(), self.get()]]
        return val

    def get(self):
        return 0

    def reset(self):
        CACHE_UNISTAT_METRICS.set(self.name, 0)


class RedisCounter(BaseRedisMetric):
    def increase(self):
        try:
            CACHE_UNISTAT_METRICS.incr(self.name)
        except Exception:
            logger.exception('Exception during incrementing')

    def produce_unistat_metric(self):
        val = [[self.get_unistat_name(), self.get()]]
        self.reset()
        return val

    def get(self) -> int:
        try:
            value = int(CACHE_UNISTAT_METRICS.get(self.name, 0))
        except TypeError:
            value = 0
        return value


class RedisSetter(BaseRedisMetric):
    def set(self, value: float):
        try:
            CACHE_UNISTAT_METRICS.set(self.name, value)
        except Exception:
            logger.exception('Exception during setting value')

    def get(self) -> float:
        try:
            value = float(CACHE_UNISTAT_METRICS.get(self.name))
        except TypeError:
            value = 0

        return value
