import logging
from datetime import datetime
from enum import Enum

from django.conf import settings
from django.utils import timezone

from wiki.sync.connect.logic import is_org_imported
from wiki.sync.connect.models import Organization
from wiki.unistat.dao import get_edit_events, get_pages, get_sync_stat
from wiki.utils.blackbox import get_oauth_token_expire_time, ExpiredTokenError, BlackboxError


logger = logging.getLogger(__name__)


UNLIMITED_OAUTH_TOKEN_TTL_VALUE = 10000


class OauthToken(Enum):
    OAUTH_TOKEN = 'OAUTH_TOKEN'


def median(array):
    """
    Calculate median of the given list.
    """
    array = sorted(array)
    half, odd = divmod(len(array), 2)
    if odd:
        return array[half]
    return (array[half - 1] + array[half]) / 2.0


def dir_stats(sync_stats):
    NOW = timezone.now()

    dir_org_ids = set(Organization.objects.all().values_list('dir_id', flat=True))

    imported_all_time = len(dir_org_ids)

    imported_lately = 0
    import_failed_lately = 0
    import_failed_alltime = 0
    sync_failed_lately = 0
    sync_failed_all_time = 0
    sync_durations_secs = []

    for sync in sync_stats:
        if sync.modified > NOW - settings.SYNC_DURATION_THRESHOLD:
            sync_durations_secs.append(sync.last_pull_duration.seconds)

        if sync.last_pull_status == sync.PULL_STATUSES.failed:
            if is_org_imported(sync, dir_org_ids):
                sync_failed_all_time += 1
            else:
                import_failed_alltime += 1
            if sync.modified > NOW - settings.SYNC_LATELY_THRESHOLD:
                if is_org_imported(sync, dir_org_ids):
                    sync_failed_lately += 1
                else:
                    import_failed_lately += 1
        else:
            if sync.modified > NOW - settings.SYNC_LATELY_THRESHOLD:
                if sync.successful_attempts == 1:
                    imported_lately += 1

    if sync_durations_secs:
        sync_duration_med = median(sync_durations_secs)
        sync_durations_avg = sum(sync_durations_secs) * 1.0 / len(sync_durations_secs)
    else:
        sync_duration_med = sync_durations_avg = 0

    return {
        # организации которые только проимпортированы, не синкались
        'imported_lately': imported_lately,
        # организации которые проимпортированы или засинканы
        'imported_alltime': imported_all_time,
        'sync_duration_med': sync_duration_med,
        'sync_durations_avg': sync_durations_avg,
        # организации которые проимпортированы но не засинканы
        'sync_failed_lately': sync_failed_lately,
        'sync_failed_alltime': sync_failed_all_time,
        # организации которые не проимпортированы
        'import_failed_lately': import_failed_lately,
        'import_failed_alltime': import_failed_alltime,
    }


def pages_stat(statuses):
    active, deleted = statuses
    return {
        'active_alltime': active,
        'deleted_alltime': deleted,
    }


def events_stat(edits_count):
    return {
        'edits_alltime': edits_count,
    }


def get_unistat_olap():
    result = []
    if settings.IS_BUSINESS:
        for key, value in dir_stats(get_sync_stat()).items():
            result.append(['directory_organisations_' + key + '_ahhh', float(value)])
    for key, value in pages_stat(get_pages()).items():
        result.append(['pages_' + key + '_ahhh', float(value)])
    for key, value in events_stat(get_edit_events().count()).items():
        result.append(['events_' + key + '_ahhh', float(value)])
    return result


def get_oauth_token_ttl_data():
    data = {}
    for token in OauthToken:
        token_value = getattr(settings, token.value)
        if not token_value:
            continue

        try:
            dt = get_oauth_token_expire_time(token_value)
            if dt:
                timedelta = dt - datetime.now()
                days = timedelta.days if timedelta.days > 0 else 0
            else:
                days = UNLIMITED_OAUTH_TOKEN_TTL_VALUE
        except ExpiredTokenError:
            days = 0
        except BlackboxError as err:
            logger.error(f'Can\'t get data from Blackbox: {repr(err)}')
            continue

        data[token.value] = days

    return data
