import os
import time
import socket
import json


def maybe_float_size(value, divider):
    """
    RTFS.
    Convenience shorthand.
    """
    if value is None:
        return None
    return float(value) / divider


def group(lst, cls=dict):
    res = cls()
    for key, val in lst:
        try:
            group_list = res[key]
        except KeyError:
            res[key] = [val]
        else:
            group_list.append(val)
    return res


def _collect_uwsgi_info(path, encoding='utf-8'):
    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    sock.connect(path)
    try:
        data = ''
        while True:
            buf = sock.recv(64 * 2 ** 10)
            if not buf:
                break
            data += buf.decode(encoding)
    finally:
        sock.close()

    data = json.loads(data)
    return data


def uwsgi_collect_sensors(data, add_sensor):
    """
    https://github.com/wdtinc/uwsgi-cloudwatch/blob/master/uwsgi_cloudwatch/main.py#L79
    Сигналы
    unistat-uwsgi_(имя)_префикс

    load_max
    signal_queue_max
    sockets_queue_total_max
    workers_total_max
    uwsgi_workers_busy_ammx - всего воркеров занято,  при роллапе - макс значение в период
    uwsgi_workers_idle_ammn - всего воркеров свободно,  при роллапе - минимальное значение в период
    unistat-uwsgi_max_avg_rt_sec_max - AverageResponseTimeMilliseconds
    longest_current_req_sec_max - длительный запрос
    """
    now_ts = time.time()

    add_sensor('load', data['load'])
    add_sensor('signal_queue', data['signal_queue'])
    add_sensor('sockets_queue_total', sum(item['queue'] for item in data['sockets']))
    add_sensor('workers_total', len(data['workers']))
    w_by_s = group((item['status'], item) for item in data['workers'])

    for key in ('idle', 'busy', 'cheap'):
        w_by_s.setdefault(key, [])

    for status, items in w_by_s.items():
        add_sensor(
            'workers_{}'.format(status),
            len(items),
            suffixes=['_ammx'],
        )
    add_sensor('workers_idle', len(w_by_s['idle']), suffixes=['_ammn'])

    avg_rts = [worker['avg_rt'] for worker in data['workers']]
    avg_rts = [val for val in avg_rts if val]
    if avg_rts:
        add_sensor('max_avg_rt_sec', float(max(avg_rts)) / 1e6)  # AverageResponseTimeMilliseconds

    cores = [core for worker in data['workers'] for core in worker['cores']]
    running_cores = [core for core in cores if core.get('req_info')]
    if running_cores:
        min_request_start = min(core['req_info'].get('request_start') for core in running_cores)
        add_sensor('longest_current_req_sec', now_ts - min_request_start)


def process_uwsgi_data_for_unistat(data, common_prefix=None):
    results = []

    # # Qloud resource aggregates, for comparison:
    # memory: thhh tmmv tvvv txxx
    # cpu usage: hgram tmmv tvvv txxx
    # cpu limit: thhh tmmv txxx
    def add_sensor(label, value, suffixes=('_max',)):
        for suffix in suffixes:
            name = '{}{}{}'.format(common_prefix, label.lower(), suffix)
            results.append([name, value])

    uwsgi_collect_sensors(data, add_sensor)

    return results


def _get_uwsgi_data(sock_path=None):
    if sock_path is None:
        sock_path = os.environ.get('UWSGI_STATS')
    if not sock_path:
        raise Exception('UWSGI_STATS socket path env variable is not set')

    data = _collect_uwsgi_info(sock_path)
    return data


def uwsgi_unistat(sock_path=None, common_prefix='uwsgi_'):
    data = _get_uwsgi_data(sock_path=sock_path)
    return process_uwsgi_data_for_unistat(data, common_prefix=common_prefix)
