from enum import Enum
from ninja import Schema
from pydantic import conint

MB = 1024 * 1024

LIMIT_UPLOAD_SESSIONS_PER_USER = 10

PartNumber = conint(ge=1, le=10000)


class Chunk(Schema):
    etag: str
    part_number: PartNumber
    size: int

    def to_aws_s3(self):
        return {'ETag': self.etag, 'PartNumber': self.part_number}

    def serialize(self):
        return {'t': self.etag, 'n': self.part_number, 's': self.size}

    @classmethod
    def deserialize(cls, o):
        return cls(
            part_number=o['n'],
            etag=o['t'],
            size=o['s'],
        )


class UploadSessionTargetType(str, Enum):
    IMPORT_PAGE = 'import_page'
    IMPORT_GRID = 'import_grid'
    ATTACHMENT = 'attachment'


class UploadSessionStatusType(str, Enum):
    NOT_STARTED = 'not_started'
    IN_PROGRESS = 'in_progress'
    FINISHED = 'finished'
    ABORTED = 'aborted'
    USED = 'used'
    MARKED_FOR_CLEANUP = 'cleanup'


TARGET_LIMITS_MAP = {
    UploadSessionTargetType.IMPORT_PAGE: 2 * MB,
    UploadSessionTargetType.IMPORT_GRID: 100 * MB,
    UploadSessionTargetType.ATTACHMENT: 100 * MB,
}
