from datetime import datetime
from datetime import timedelta
from django.contrib.auth import get_user_model
from django.utils import timezone

from wiki.api_v2.exceptions import AlreadyExists
from wiki.api_v2.public.upload_sessions.exceptions import (
    UploadSessionFileSizeLimit,
    UploadSessionLimitExceeded,
)

from wiki.api_v2.public.upload_sessions.schemas import CreateUploadSessionSchema
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.uploads.consts import LIMIT_UPLOAD_SESSIONS_PER_USER
from wiki.uploads.consts import TARGET_LIMITS_MAP
from wiki.uploads.consts import UploadSessionStatusType
from wiki.uploads.models import UploadSession
from wiki.uploads.s3_client import S3_CLIENT

User = get_user_model()


def set_in_progress(upload_session: UploadSession):
    upload_session.status = UploadSessionStatusType.IN_PROGRESS
    upload_session.save()


def set_finished(upload_session: UploadSession):
    upload_session.status = UploadSessionStatusType.FINISHED
    upload_session.finished_at = datetime.now()
    upload_session.save()


def check_limit_sessions(user: User, org: BaseOrganization):
    exclude_statuses = [
        UploadSessionStatusType.FINISHED,
        UploadSessionStatusType.ABORTED,
        UploadSessionStatusType.USED,
        UploadSessionStatusType.MARKED_FOR_CLEANUP,
    ]
    uploads_counts = (
        UploadSession.objects.filter(user=user, org=org.as_django_model())
        .exclude(status__in=exclude_statuses)
        .count()
    )

    if uploads_counts >= LIMIT_UPLOAD_SESSIONS_PER_USER:
        raise UploadSessionLimitExceeded


def check_limit_size(data: CreateUploadSessionSchema):
    if data.file_size >= TARGET_LIMITS_MAP[data.target]:
        raise UploadSessionFileSizeLimit


def check_session_non_exists(user: User, org: BaseOrganization, data: CreateUploadSessionSchema):
    try:
        (
            UploadSession.objects.get(
                created_at=timezone.now() - timedelta(seconds=10),
                user=user,
                org=org.as_django_model(),
                file_name=data.file_name,
                file_size=data.file_size,
                target=data.target,
                status=UploadSessionStatusType.NOT_STARTED,
            )
        )
    except UploadSession.DoesNotExist:
        return

    raise AlreadyExists


def create_upload_session(user: User, organization: BaseOrganization, data: CreateUploadSessionSchema) -> UploadSession:
    upload_session = UploadSession(
        target=data.target,
        file_name=data.file_name,
        file_size=data.file_size,
        user=user,
        created_at=timezone.now(),
        org=organization.as_django_model(),
    )
    upload_session.upload_id = S3_CLIENT.create_multipart_upload(upload_session.storage_key)
    upload_session.status = UploadSessionStatusType.IN_PROGRESS
    upload_session.save()

    return upload_session
