import uuid
from typing import List

from django.conf import settings
from django.contrib.postgres.fields import JSONField
from django.db import models
from enumfields import EnumField

from wiki.sync.connect.models import Organization
from wiki.uploads.consts import UploadSessionTargetType, UploadSessionStatusType, Chunk
from wiki.uploads.s3_client import S3_CLIENT


class UploadSession(models.Model):
    objects: models.Manager
    session_id = models.UUIDField(primary_key=True, default=uuid.uuid4)

    target = EnumField(UploadSessionTargetType, max_length=11)

    file_name = models.CharField(max_length=255)
    upload_id = models.CharField(max_length=255, null=True, blank=True)

    file_size = models.PositiveIntegerField(default=0)
    status = EnumField(UploadSessionStatusType, default=UploadSessionStatusType.NOT_STARTED, max_length=11)

    user = models.ForeignKey(settings.AUTH_USER_MODEL, on_delete=models.CASCADE)
    org = models.ForeignKey(Organization, blank=True, null=True, default=None, on_delete=models.CASCADE)

    created_at = models.DateTimeField(auto_now_add=True)
    finished_at = models.DateTimeField(null=True, blank=True)

    chunks = JSONField(blank=True, null=True)

    @property
    def storage_key(self):
        created = self.created_at.strftime('%Y-%m-%d')
        return f'{settings.WIKI_CODE}/{self.target.value}/{self.user_id}/{created}/{self.session_id}'

    @property
    def uploaded_size(self):
        return sum([c.size for c in self.get_chunks()])

    def get_chunks(self) -> List[Chunk]:
        if self.chunks:
            return [Chunk.deserialize(o) for o in self.chunks]

        return []

    def add_chunk(self, c: Chunk):
        chunks = {c.part_number: c for c in self.get_chunks()}
        chunks[c.part_number] = c
        self.chunks = [q.serialize() for q in chunks.values()]

    @property
    def data_stream(self):
        return S3_CLIENT.get_object_body_stream(self.storage_key)
