import uuid
from typing import List

import boto3
from botocore.exceptions import ClientError
from django.conf import settings

from wiki.api_v2.public.upload_sessions.exceptions import StorageUploadError
from wiki.uploads.consts import Chunk, PartNumber

MOCKED_STORAGE_HOST = 'https://mocked-storage.yandex-team.ru/'


class BaseS3Client:
    is_mock: bool = True

    def create_bucket(self):
        pass

    def complete_multipart_upload(self, key: str, upload_id: str, chunks: List[Chunk]):
        pass

    def upload_part(self, body: bytes, key: str, part_number: PartNumber, upload_id: str) -> str:
        return f'{upload_id}-{part_number}'

    def abort_multipart_upload(self, key: str, upload_id: str):
        pass

    def create_multipart_upload(self, key: str) -> str:
        return str(uuid.uuid4())

    def key_exists(self, key: str) -> bool:
        return True

    def delete_objects(self, keys: List[str]):
        pass

    def get_object_body_stream(self, key: str):
        class Body:
            def iter_chunks(self, chunk_size):
                return []

            def read(self):
                return b''
        return Body()

    def put_object(self, key, body):
        pass

    def generate_presigned_url(self, key: str, expires_in: int, response_headers: dict = None):
        return f'{MOCKED_STORAGE_HOST}{key}'


class S3Client(BaseS3Client):
    is_mock: bool = False

    def __init__(self, bucket):
        self.bucket = bucket

    def _get_client(self):
        return boto3.client(
            service_name='s3',
            aws_access_key_id=settings.S3_ACCESS_KEY,
            aws_secret_access_key=settings.S3_ACCESS_SECRET_KEY,
            endpoint_url=settings.S3_ENDPOINT_URL,
            use_ssl=False,
            verify=False,
        )

    def complete_multipart_upload(self, key: str, upload_id: str, chunks: List[Chunk]):
        try:
            self._get_client().complete_multipart_upload(
                Bucket=self.bucket,
                Key=key,
                MultipartUpload={'Parts': [chunk.to_aws_s3() for chunk in sorted(chunks, key=lambda x: x.part_number)]},
                UploadId=upload_id,
            )
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def abort_multipart_upload(self, key: str, upload_id: str):
        try:
            self._get_client().abort_multipart_upload(
                Bucket=self.bucket,
                Key=key,
                UploadId=upload_id,
            )
        except ClientError as err:
            if err.response['Error']['Code'] == 'NoSuchUpload':
                return
            raise StorageUploadError.from_aws(err)

    def upload_part(self, body: bytes, key: str, part_number: PartNumber, upload_id: str) -> str:
        try:
            response = self._get_client().upload_part(
                Body=body,
                Bucket=self.bucket,
                ContentLength=len(body),
                Key=key,
                PartNumber=part_number,
                UploadId=upload_id,
            )
            return response['ETag']
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def create_bucket(self):
        self._get_client().create_bucket(Bucket=self.bucket)

    def create_multipart_upload(self, key: str) -> str:
        try:
            response = self._get_client().create_multipart_upload(
                Bucket=self.bucket,
                Key=key,
            )
            return response['UploadId']
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def key_exists(self, key: str) -> bool:
        try:
            self._get_client().head_object(
                Bucket=self.bucket,
                Key=key,
            )
            return True
        except ClientError as err:
            if err.response.get('Error', {}).get('Code') == 404:
                return False
            raise StorageUploadError.from_aws(err)

    def delete_objects(self, keys: List[str]):
        try:
            self._get_client().delete_objects(
                Bucket=self.bucket,
                Delete={'Objects': [{'Key': key} for key in keys]},
            )
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def get_object_body_stream(self, key: str):
        try:
            response = self._get_client().get_object(Bucket=self.bucket, Key=key)
            return response['Body']
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def put_object(self, key, body):
        try:
            self._get_client().put_object(Bucket=self.bucket, Key=key, Body=body)
        except ClientError as err:
            raise StorageUploadError.from_aws(err)

    def generate_presigned_url(self, key: str, expires_in: int = 60, response_headers: dict = None):
        response_headers_map = {
            'cache-control': 'ResponseCacheControl',
            'content-disposition': 'ResponseContentDisposition',
            'content-encoding': 'ResponseContentEncoding',
            'content-language': 'ResponseContentLanguage',
            'content-type': 'ResponseContentType',
            'expires': 'ResponseExpires',
        }
        try:
            params = {'Bucket': self.bucket, 'Key': key}
            if response_headers:
                for header in response_headers:
                    l_header = header.lower()
                    if l_header in response_headers_map:
                        header_key = response_headers_map[l_header]
                        params[header_key] = response_headers[header]

            return self._get_client().generate_presigned_url(
                ClientMethod='get_object',
                Params=params,
                ExpiresIn=expires_in,
            )
        except ClientError as err:
            raise StorageUploadError.from_aws(err)


S3_CLIENT: BaseS3Client = BaseS3Client() if settings.USE_S3_MOCK else S3Client(settings.S3_USER_UPLOADS_BUCKET)
