import logging
from datetime import datetime, timedelta

from django.db.models import Q
from django.utils import timezone

from wiki.uploads.consts import UploadSessionStatusType
from wiki.uploads.models import UploadSession
from wiki.uploads.s3_client import S3_CLIENT
from wiki.utils.tasks.base import LockedCallableTask

logger = logging.getLogger(__name__)


class ClearUploadSessions(LockedCallableTask):
    """
    Удалить неиспользуемые UploadSession и файлы в S3
    """

    name = 'wiki.clear_upload_sessions'
    logger = logging.getLogger(name)
    time_limit = 60 * 60  # 60 минут

    def run(self, *args, **kwargs):
        date = timezone.now()
        clear_upload_sessions(date)


def clear_upload_sessions(date: datetime):
    query = Q(status=UploadSessionStatusType.IN_PROGRESS, created_at__lte=date - timedelta(hours=12))
    query |= Q(status=UploadSessionStatusType.NOT_STARTED, created_at__lte=date - timedelta(days=1))
    query |= Q(status=UploadSessionStatusType.ABORTED, created_at__lte=date - timedelta(days=1))
    query |= Q(status=UploadSessionStatusType.FINISHED, finished_at__lte=date - timedelta(days=1))
    query |= Q(status=UploadSessionStatusType.MARKED_FOR_CLEANUP)
    query |= Q(status=UploadSessionStatusType.USED)

    upload_sessions = list(UploadSession.objects.filter(query))

    # оканчиваем загрузку файлов
    abort_upload = (session for session in upload_sessions if session.status == UploadSessionStatusType.IN_PROGRESS)
    for session in abort_upload:
        S3_CLIENT.abort_multipart_upload(session.storage_key, upload_id=session.upload_id)
        logger.info(f'S3. abort old upload. key=`{session.storage_key}`, upload_id=`{session.upload_id}`')

    # удаляем файлы на S3
    statuses_for_deleting_files = {UploadSessionStatusType.FINISHED, UploadSessionStatusType.MARKED_FOR_CLEANUP}
    delete_files = [session for session in upload_sessions if session.status in statuses_for_deleting_files]

    if delete_files:
        delete_keys = [session.storage_key for session in delete_files]
        S3_CLIENT.delete_objects(delete_keys)
        logger.info('S3. delete old objects. keys=[{}]'.format(', '.join(delete_keys)))

    # удаляем модели
    if upload_sessions:
        session_ids = [session.session_id for session in upload_sessions]
        UploadSession.objects.filter(session_id__in=session_ids).delete()
        logger.info(f'S3. delete old models. count={len(session_ids)}')
