from typing import Optional

from django.contrib.auth.models import AbstractUser
from django.db import models
from jsonfield import JSONField

from wiki.acl.consts import AclSubject, AclSubjectType


class AbstractWikiUser(AbstractUser):
    """
    Абстрактная модель Вики пользователя.
    Наследует все свойства дефолтной модели пользователя auth.User, расширяя ее специфичными для профайла
    Вики пользователя свойствами.
    Содержит общие для всех типов Вики методы.
    """

    profile = JSONField(max_length=4000, default={})

    # значение атрибута has_personal_cluster актуально только для пользователей, состоящих в одной организации,
    # поэтому не может быть валидным для B2B пользователей
    has_personal_cluster = models.BooleanField(default=False)

    # -- на будущее, когда все пользователи получат облачный UID --

    def set_cloud_uid(self, cloud_uid: str):
        pass

    def get_cloud_uid(self) -> Optional[str]:
        return None

    @classmethod
    def find_by_uid(cls, uid):
        raise cls.DoesNotExist()

    def set_uid(self, cloud_uid: str):
        pass

    def get_uid(self) -> str:
        return None

    # --------------------------------------------------------

    def is_robot(self):
        from wiki.users.logic import is_robot

        return is_robot(self)

    def is_admin(self):
        from wiki.pages.access import is_admin

        return is_admin(self)

    def set_user_setting(self, name, value):
        """
        Сохраняет в настройках пользователя настройку с именем name и значением value.
        """
        from wiki.users.logic import set_user_setting

        return set_user_setting(self, name, value)

    def get_acl_subject(self):
        return AclSubject(subj_id=self.id, subj_type=AclSubjectType.USER)

    class Meta:
        db_table = 'auth_user'
        abstract = True
