
from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction

from wiki.intranet.models import Staff
from wiki.org import get_org
from wiki.utils import timezone


def create_user_and_staff(**params):
    user_params = [
        'username',
        'first_name',
        'last_name',
        'email',
    ]

    msg = 'Not all required params given for user creation'
    assert all(key in params for key in user_params), msg

    User = get_user_model()
    user = User(
        username=params['username'],
        first_name=params['first_name'],
        last_name=params['last_name'],
        email=params['email'],
        password='!',
    )

    assert 'uid' in params, 'uid required for staff creation'
    create_staff(user=user, uid=params['uid'], lang=params.get('lang'))

    return user


def create_staff(user, uid, lang=None):
    """
    user должен быть несохраненный, иначе будет
    проблема при создании кластера без стаффа.
    """
    now = timezone.now()

    login_ld = (
        '%s_%s' % (user.username, get_org().dir_id)
        if (settings.IS_BUSINESS and not settings.IS_TESTS)
        else user.username
    )

    staff = Staff(
        login=user.username,
        login_ld=login_ld,
        uid=uid,
        first_name=user.first_name,
        last_name=user.last_name,
        work_email=user.email,
        lang_ui=lang or 'ru',
        created_at=now,
        modified_at=now,
        # Имя пользователя в интерфейсе отображается из wiki_name
        wiki_name=make_wiki_name(user),
    )

    with transaction.atomic():
        # staff должен быть сохранен первым,
        # иначе сигнал при сохранении пользователя не может создать кластер
        # После этого staff должен быть еще раз пересохранен,
        # чтобы он связался с пользователем
        staff.save()
        user.save()
        staff.user = user
        staff.save()

    return staff


def make_wiki_name(user):
    return user.first_name.lower().capitalize() + user.last_name.lower().capitalize()


def get_support_contact():
    """
    Возвращает либо значение SUPPORT_CONTACT из настроек либо список staff ссылок с логинами администраторов организации
    для бизнесс Вики.


    """
    if settings.IS_BUSINESS:
        # эта функция дергается при составлении сообщения об ошибке.
        # А если ошибка была порождена тем что у нас нет такой организации, тут мы упадем еще разок
        # и все посыпется в stderr вместо нормальных логгеров
        try:
            from wiki.users_biz.dao import get_admin_group_members

            return ', '.join([admin.email for admin in get_admin_group_members()[:15]])
        except Exception:  # noqa
            return '-'
    else:
        return settings.SUPPORT_CONTACT
