from typing import List, Iterable

from django.conf import settings
from django.contrib.auth import get_user_model
from django.core.exceptions import ObjectDoesNotExist
from django.db.models import Q  # noqa
from django.utils.translation import ugettext_lazy as _
from wiki.api_core.errors.bad_request import InvalidDataSentError
from wiki.api_frontend.serializers.group_identity import GroupIdentity
from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.org import org_staff, org_user  # noqa
from wiki.sync.connect.base_organization import BaseOrganization
from wiki.utils.features.get_features import get_wiki_features

if settings.IS_BUSINESS:
    from wiki.users_biz.dao import *  # noqa
else:
    from wiki.users_ynx.dao import *  # noqa


class UserNotFound(Exception):
    def __init__(self, missing_idx=None):
        self.missing_identities = missing_idx or []


class GroupNotFound(Exception):
    pass


class ConfigurationMismatch(Exception):
    pass


def get_group_by_identity(identity: GroupIdentity, organization: BaseOrganization):
    try:
        return organization.get_groups().get(pk=identity.id)
    except ObjectDoesNotExist:
        pass


def get_groups_by_identity(
    identities: List[GroupIdentity], organization: BaseOrganization, panic_if_missing: bool = True
):
    try:
        grps = list(organization.get_groups().filter(pk__in=[identity.id for identity in identities]))
        if panic_if_missing and len(grps) != len(identities):
            raise GroupNotFound
    except ObjectDoesNotExist:
        pass


def get_user_by_identity(
    identity: UserIdentity, prefetch_staff=False, limit_users_by_current_org=True, organization: BaseOrganization = None
):
    if organization:
        mdl = organization.get_users()
    else:
        if limit_users_by_current_org:
            mdl = org_user()
        else:
            mdl = get_user_model().objects

    if prefetch_staff:
        mdl = mdl.select_related('staff')

    if identity.cloud_uid and get_wiki_features().cloud_uid_supported:
        try:
            return mdl.get(cloud_uid=identity.cloud_uid)
        except ObjectDoesNotExist:
            pass

    if identity.uid:
        try:
            return mdl.get(staff__uid=identity.uid)
        except ObjectDoesNotExist:
            pass

    raise UserNotFound([identity])


def get_users_by_identity(
    identities: List[UserIdentity],
    prefetch_staff=False,
    panic_if_missing=True,
    organization: BaseOrganization = None,
    apiv1_exceptions: bool = True,
    preserve_order: bool = False,
):
    if identities is None:
        return []

    identities = list(dict.fromkeys(identities))

    users = _get_users_by_identity(identities, prefetch_staff, organization=organization)

    if len(users) != len(identities) and panic_if_missing:
        found_identities = set([UserIdentity.from_user(u) for u in users])
        missing_identities = set(identities) - found_identities
        if apiv1_exceptions:
            raise InvalidDataSentError(
                # Translators:
                #  ru: Пользователи с UID "%s" не найдены
                #  en: Users with UIDS "%s" not found
                _('Users with UIDS "%s" not found')
                % ','.join([str(m) for m in missing_identities])
            )
        else:
            raise UserNotFound(missing_identities)

        pass

    if preserve_order:
        priority_by_uid = {identity.uid: idx + 1 for idx, identity in enumerate(identities) if identity.uid is not None}
        priority_by_cloud_uid = {
            identity.cloud_uid: idx + 1 for idx, identity in enumerate(identities) if identity.cloud_uid is not None
        }

        users = sorted(
            users,
            key=lambda user: priority_by_uid.get(user.get_uid(), None)
            or priority_by_cloud_uid.get(user.get_cloud_uid(), None)
            or 0,
        )

    return users


def _get_users_by_identity(identities: Iterable[UserIdentity], prefetch_staff=False, organization=None):
    if organization:
        mdl = organization.get_users()
    else:
        mdl = org_user()

    if prefetch_staff:
        mdl = mdl.select_related('staff')

    uids = [i.uid for i in identities if i.uid is not None]
    cloud_uids = [
        i.cloud_uid for i in identities if i.cloud_uid is not None and get_wiki_features().cloud_uid_supported
    ]

    if len(uids) == 0 and len(cloud_uids) == 0:
        return []

    if len(cloud_uids) > 0 and len(uids) == 0:
        return mdl.filter(cloud_uid__in=cloud_uids)

    if len(uids) > 0 and len(cloud_uids) == 0:
        return mdl.filter(staff__uid__in=uids)

    return mdl.filter(Q(cloud_uid__in=cloud_uids) | Q(staff__uid__in=uids))


def staff_model_by_login(login):
    """
    Вернуть Staff-профиль пользователя по login.

    @param login: basestring
    """
    return org_staff().get(login=login)
