
import logging

from blackbox import Blackbox
from django import forms
from django.conf import settings
from django.contrib.admin.widgets import FilteredSelectMultiple
from django.contrib.auth import get_user_model
from django.contrib.auth.forms import UserChangeForm, UserCreationForm
from django.core.validators import MaxLengthValidator, RegexValidator
from django.db import transaction
from django.db.models import Q
from django.utils.translation import ugettext as _
from wiki.intranet.models import Staff

from wiki.users.core import create_staff, make_wiki_name
from wiki.users.models import Group
from wiki.utils import timezone
from wiki.utils.tvm2 import get_service_ticket

User = get_user_model()

blackbox = Blackbox(url=settings.BLACKBOX_URL)
logger = logging.getLogger('django.request')


class AddUserForm(UserCreationForm):
    """
    Форма для ручного создания пользователя.
    Используется во внешних инстансах вик
    """

    def __init__(self, *args, **kwargs):
        # Админка добавляет username в exclude
        try:
            self._meta.exclude.remove('username')
        except (ValueError, AttributeError):
            pass

        self.base_fields['email'].required = True
        self.base_fields['first_name'].required = True
        self.base_fields['last_name'].required = True
        self.base_fields['password1'].required = False
        self.base_fields['password2'].required = False
        self.base_fields['username'].regex = r'^[\w.+-]+$'
        self.base_fields['username'].help_text = _(
            'Required. 30 characters or fewer. ' 'Letters, digits and /./+/-/_ only.'
        )
        if settings.IS_BUSINESS:
            # отключаем валидацию, так как это мешает нам импортировать данные из Директории
            # и увеличиваем лимит длины поля username
            self.base_fields['email'].validators = list()
            username_field = self.base_fields['username']
            for val in username_field.validators:
                if isinstance(val, MaxLengthValidator):
                    val.limit_value = 200
                if isinstance(val, RegexValidator):
                    username_field.validators.remove(val)
        super(AddUserForm, self).__init__(*args, **kwargs)

    class Meta:
        model = User
        exclude = ()

    def clean_password(self):
        """
        Нам совсем не нужен пароль джанго-пользователя
        """
        return '!'

    def clean_username(self):
        username = self.cleaned_data['username'].lower()
        uid = self.data.get('uid')

        if uid:
            blackbox_uid = uid
        else:
            blackbox_uid = blackbox.uid(
                username, headers={'X-Ya-Service-Ticket': get_service_ticket(settings.TVM2_BLACKBOX_CLIENT.value)}
            )
            if not blackbox_uid:
                raise forms.ValidationError(_('personalisation:NotFoundAtBlackbox'))

        self.blackbox_uid = blackbox_uid

        # Проверяем существование пользователя по нормализованному логину и уиду
        if settings.IS_BUSINESS:
            # в бизнесе, в отличии от интранета, логин пользователя не является уникальным,
            # поэтому проверяем уникальность только по uid
            condition = Q(uid=blackbox_uid)
        else:
            condition = Q(normal_login=username) | Q(uid=blackbox_uid)
        if Staff.objects.filter(condition).exists():
            raise forms.ValidationError(_('personalisation:UserAlreadyExists'))

        return username

    def validate_unique(self):
        if settings.IS_BUSINESS:
            # Для того, чтобы не ругался на неуникальность поля username.
            return
        super(AddUserForm, self).validate_unique()

    def save(self, commit=True):
        user = super(UserCreationForm, self).save(commit=False)
        user.set_unusable_password()

        create_staff(user=user, uid=self.blackbox_uid)

        return user


class EditUserForm(UserChangeForm):
    """
    Форма для ручного редактирования пользователя.
    Используется во внешних инстансах вик
    """

    def __init__(self, *args, **kwargs):
        super(EditUserForm, self).__init__(*args, **kwargs)
        self.fields['email'].required = True
        self.fields['first_name'].required = True
        self.fields['last_name'].required = True

    class Meta:
        model = User
        exclude = ['password', 'date_joined', 'last_login']

    def clean_password(self):
        """
        Нам совсем не нужен пароль джанго-пользователя
        """
        return '!'

    def validate_unique(self):
        if settings.IS_BUSINESS:
            # Для того, чтобы не ругался на неуникальность поля username.
            return
        super(EditUserForm, self).validate_unique()

    def save(self, commit=True):
        user = super(EditUserForm, self).save(commit=False)

        try:
            staff = user.staff
        except (Staff.DoesNotExist, Staff.MultipleObjectsReturned):
            logger.exception('User "%s" does not have a profile', user.username)
            raise

        staff.first_name = user.first_name
        staff.last_name = user.last_name
        staff.work_email = user.email
        staff.modified_at = timezone.now()
        staff.wiki_name = make_wiki_name(user)

        with transaction.atomic():
            # Параметр commit=False игнорируется, чтобы избежать разъехавшихся данных
            staff.save()
            user.save()

        return user


class EditGroupForm(forms.ModelForm):
    users = forms.ModelMultipleChoiceField(
        queryset=User.objects.all(),
        widget=FilteredSelectMultiple(
            verbose_name=_('Users'),
            is_stacked=False,
        ),
        required=False,
    )

    class Meta:
        model = Group
        exclude = ()

    def __init__(self, *args, **kwargs):
        instance = kwargs.get('instance', None)
        if instance is not None:
            initial = kwargs.get('initial', {})
            initial['users'] = instance.user_set.all()
            kwargs['initial'] = initial
        super(EditGroupForm, self).__init__(*args, **kwargs)

    def save(self, commit=True):
        group = super(EditGroupForm, self).save(commit=commit)

        # Если commit=True, то можно сразу сохранить m2m-поля.
        # Если commit=False, то объекта может не быть в базе,
        # поэтому сохранение полей выносится в пропатченный save_m2m.
        # При этом save_m2m становится unbound-методом, и self туда передается из замыкания.
        if commit:
            group.user_set.set(self.cleaned_data['users'])
        else:
            old_save_m2m = self.save_m2m

            def new_save_m2m():
                old_save_m2m()
                group.user_set.set(self.cleaned_data['users'])

            self.save_m2m = new_save_m2m
        return group
