from django.contrib.auth import get_user_model
from django.db import IntegrityError, transaction

from wiki.api_v2.public.me.exceptions import InvalidStatusTransition, UserFeatureIsProcessing
from wiki.users.models import UserFeature
from wiki.users.consts import UserFeatureCode, UserFeatureStatus
from wiki.users.logic.settings import set_user_setting
from wiki.users.tasks import ProcessUserFeature

status_transitions = {
    UserFeatureStatus.ENABLED: {UserFeatureStatus.PENDING_DISABLE, UserFeatureStatus.PENDING_ENABLE},
    UserFeatureStatus.PENDING_ENABLE: {UserFeatureStatus.ENABLED, UserFeatureStatus.PENDING_DISABLE},
    UserFeatureStatus.DISABLED: {UserFeatureStatus.PENDING_ENABLE, UserFeatureStatus.PENDING_DISABLE},
    UserFeatureStatus.PENDING_DISABLE: {UserFeatureStatus.DISABLED, UserFeatureStatus.PENDING_ENABLE},
}

feature_settings = {
    UserFeatureCode.DATA_UI_WEB: 'data_ui_web',
}

User = get_user_model()


def enable_feature(user: User, feature_code: UserFeatureCode):
    _switch_status(user, feature_code, UserFeatureStatus.PENDING_ENABLE)
    if setting_name := feature_settings.get(feature_code, None):
        set_user_setting(user, setting_name, True)


def disable_feature(user: User, feature_code: UserFeatureCode):
    _switch_status(user, feature_code, UserFeatureStatus.PENDING_DISABLE)
    if setting_name := feature_settings.get(feature_code, None):
        set_user_setting(user, setting_name, False)


@transaction.atomic
def _switch_status(user: User, feature_code: UserFeatureCode, status: UserFeatureStatus) -> bool:
    feature = UserFeature.objects.select_for_update().filter(user=user, feature_code=feature_code).first()

    if feature:
        if feature.status == status:
            return True

        if status not in status_transitions.get(feature.status, set()):
            raise InvalidStatusTransition(debug_message=f'Invalid transition from {feature.status} to {status}')

        feature.status = status
        feature.save()
        ProcessUserFeature().delay(feature_id=feature.id)
        return True

    try:
        feature = UserFeature.objects.create(
            user=user,
            feature_code=feature_code,
            status=status,
        )
        ProcessUserFeature().delay(feature_id=feature.id)
    except IntegrityError:
        pass

    return True


def assert_feature_is_not_processing(user: User, feature_code: UserFeatureCode):
    is_processing = UserFeature.objects.filter(
        user=user, feature_code=feature_code, status=UserFeatureStatus.PENDING_ENABLE
    ).exists()

    if is_processing:
        raise UserFeatureIsProcessing()
