import waffle

from django.conf import settings as django_settings
from model_utils import Choices

from wiki.api_core.waffle_switches import DEFAULT_NEW_UI
from wiki.legacy.langs import get_default_lang
from wiki.utils.errors import ValidationError

CODE_THEMES = Choices(
    ('github', 'GITHUB', 'GitHub'),
    ('idea', 'IDEA', 'IDEA'),
    ('solarized_light', 'SOLARIZED_LIGHT', 'Solarized Light'),
    ('solarized_dark', 'SOLARIZED_DARK', 'Solarized Dark'),
    ('dark', 'DARK', 'Dark'),
)

UI_THEMES = Choices(
    ('light', 'LIGHT', 'Light'),
    ('dark', 'DARK', 'Dark'),
    ('system', 'SYSTEM', 'System'),
)

SETTINGS_WITH_DEFAULTS = {
    'code_theme': CODE_THEMES.GITHUB,
    'propose_content_translation': False,
    'new_favorites': True,
    'new_subscriptions': True,
    'use_nodejs_frontend': False,
    'use_full_width_content': False,
    'use_new_wf': False,
    'ui_theme': UI_THEMES.LIGHT,
    'data_ui_web': False,
}

if not django_settings.IS_INTRANET:
    SETTINGS_WITH_DEFAULTS['language'] = 'ru'


def get_user_setting(user, name):
    """
    Возвращает текущее значение настройки с именем name в настройках пользователя.
    Если настройка не задана, возвращает значение по умолчанию для этой
    настройки, если значение задано в словаре SETTINGS_WITH_DEFAULTS.

    В случае неизвестной настройки — ValidationError.

    @type user: User
    @type name: basestring
    @rtype: object|None
    @raises ValidationError
    """
    assert not isinstance(user, str)

    _raise_if_unknown(name)
    profile = getattr(user, 'profile', {})
    value = profile.get(name)
    if value is None:
        value = SETTINGS_WITH_DEFAULTS.get(name)
    return value


def get_user_settings(user):
    assert not isinstance(user, str)

    profile = getattr(user, 'profile', {})
    settings = {}
    for setting, default in SETTINGS_WITH_DEFAULTS.items():
        if setting == 'data_ui_web' and waffle.switch_is_active(DEFAULT_NEW_UI):
            default = can_change_ui(user)

        settings[setting] = profile.get(setting, default)
    return settings


def set_user_setting(user, name, value):
    """
    Сохраняет в настройках пользователя настройку с именем name и значением value.

    @type user: User
    @type name: basestring
    @raises ValidationError
    """
    _raise_if_unknown(name)
    user.profile[name] = value
    user.save()


def update_user_settings(user, data):
    for key, value in data.items():
        _raise_if_unknown(key)
        user.profile[key] = value
    user.save()


def _raise_if_unknown(setting):
    if setting not in SETTINGS_WITH_DEFAULTS:
        raise ValidationError('Unknown setting %s' % setting)


def get_user_lang_ui(user):
    """
    @type user: User
    @rtype basestring
    """
    if user and not user.is_anonymous:
        lang_ui = user.staff.lang_ui
        if lang_ui:
            return lang_ui
    return get_default_lang()


def set_user_lang_ui(user, language):
    """
    @type user: User
    """
    if user and not user.is_anonymous:
        user.staff.lang_ui = language
        user.staff.save()


def uses_new_favorites(user):
    user_settings = get_user_settings(user)
    return user_settings.get('new_favorites', False)


def uses_new_subscriptions(user):
    user_settings = get_user_settings(user)
    return user_settings.get('new_subscriptions', False)


def set_uses_apiv2(user):
    user.profile['new_subscriptions'] = True
    user.profile['new_favorites'] = True
    user.save()


def can_change_ui(user) -> bool:
    if hasattr(user, 'is_robot') and user.is_robot():  # disabled by default for robots
        return False
    if user.username in django_settings.FORBIDDEN_USERS_TO_CHANGE_UI:
        return False
    return True
