"""
Этот модуль отвечает за туториал - обучение пользователя.

Туториал делится на стадии. Каждая стадия представляет собой
описание какой-то функциональности Вики.

Все взаимодействие с фронтендом сводится к двум ручкам:

1) После отображения страницы фронтенд спрашивает у бекенда,
   нужно ли ему отобразить какую-либо стадию обучения
   для данного пользователя. Бекенд возвращает один из
   трех возможных результатов:

   a) Отобразить стадию с именем stage

   {
     'status': 'display_stage',
     'stage': 'promo',
   }

   b) В данный момент ничего отображать не надо

   {
     'status': 'nothing_to_display',
   }

   c) Пользователь прошел весь туториал

   {
     'status': 'tutorial_completed',
   }

   Вариант c) выделен отдельно для оптимизации, чтобы на фронтенде можно было
   на какое-то время запомнить то, что пользователь прошел туториал,
   во избежание лишних запросов на бекенд. Фронтенд запоминает на какое-то
   время, а не навсегда, т.к. могут добавляться новые стадии туториала.

2) Фронтенд сообщает бекенду о том, что данный пользователь
   прошел стадию с указанным именем. Эта информация запоминается в
   профиле пользователя, вместе с датой изучения стадии (timestamp в секундах).

   user.profile['tutorial'] =
   {
     'promo': 1489141452, # <имя стадии>: <timestamp, когда изучена>
     'edit_basics': 1489162347,
     ...
   }

   У нового пользователя нет изученных стадий, поэтому для него

   user.profile['tutorial'] = {}

Для показа стадии нужно соблюдение трех условий:

1) Точки входа (EntryPoint).

   Точка входа - это некоторое состояние интерфейса пользователя.
   Например, точками входа являются отображение страницы, отображение
   виджета доступа, режим редактирования страницы и т.д.

2) Типа страницы (PageType).

   Каждая страница интерпретируется как набор ее типов, от частного к общему.
   На данный момент выделено три типа страницы:

   a) USER_CLUSTER - корневая страница пользовательского кластера, /users/<login>
   b) USER_GRID - грид, созданный пользователем
   c) ANY_PAGE - любая страница (включая гриды)

   Страница может быть проинтерпретирована как
   (USER_CLUSTER, ANY_PAGE), либо как (USER_GRID), либо как (ANY_PAGE).

   При выборе стадии предпочтение отдается стадии с более частным типом страницы.
   Например, если мы находимся в точке входа "отображение страницы", и есть две стадии

   a) Подсказка к кнопке правка, для типа страницы USER_CLUSTER
   b) Информация про комментарии, для типа страницы ANY_PAGE

   и мы находимся на странице (USER_CLUSTER, ANY_PAGE), то будет выбрана
   стадия "Подсказка к кнопке правка". Если все стадии для типа
   страницы USER_CLUSTER уже изучены, будет выбрана подсказка "Информация о комментарии".

3) Прошло достаточно времени с момента изучения последней стадии.

   Для комбинации точки входа и типа страницы (EntryPoint, PageType)
   может быть задан таймаут в секундах (timeout), который говорит о том,
   что не нужно показывать очередную стадию с такими (EntryPoint, PageType),
   если предыдущая стадия с этими (EntryPoint, PageType) была пройдена менее,
   чем timeout секунд назад.

   Этот таймаут позволяет показывать стадии туториала постепенно, а не все подряд.
   В данный момент таймаут определен для комбинации (Отображение страницы, Любая страница),
   благодаря чему пользователь постепенно обучается подписке, комментариям, сайдбару и т.д.

   В профиле пользователя для каждой стадии сохраняется момент
   времени, когда пользователь ее прошел. Если стадия удовлетворяет условиям 1) и 2),
   то проверяется, задан ли таймаут для ее пары (EntryPoint, PageType).

   Если таймаут не задан, то стадию можно показывать.

   Если таймаут задан, то из профиля пользователя извлекаются все пройденные стадии
   с такими же (EntryPoint, PageType), и берется стадия с самой поздней датой изучения.
   Если с момента времени, соответствующего этой самой поздно изученной стадии,
   еще не прошло timeout секунд, то данная стадия не должна отображаться.

У этих правил есть исключение - стадия 'promo'. Она всегда должна отображаться
первой. Поэтому, когда человек первый раз заходит на Вики, то ему отображается
"promo" независимо от типа страницы и точки входа.

"""

import operator
import time

import waffle
from django.conf import settings

from wiki.pages.models import Page


class EntryPoint:
    # Отображение страницы
    VIEW = 'view'

    # Режим редактирования страницы
    EDIT = 'edit'

    # Виджет создания страницы
    CREATE = 'create'

    # Виджет доступа
    ACCESS = 'access'

    # Виджет настроек
    SETTINGS = 'settings'

    # Виджет перемещения
    MOVE = 'move'

    # Виджет подписки
    SUBSCRIBE = 'subscribe'


ALL_ENTRY_POINTS = (
    EntryPoint.VIEW,
    EntryPoint.EDIT,
    EntryPoint.CREATE,
    EntryPoint.ACCESS,
    EntryPoint.SETTINGS,
    EntryPoint.MOVE,
    EntryPoint.SUBSCRIBE,
)


class PageType:
    # Корневая страница пользовательского кластера, /users/<login>
    USER_CLUSTER = 'user_cluster'

    # Грид, созданный пользователем
    USER_GRID = 'user_grid'

    # Любая страница (включая гриды)
    ANY_PAGE = 'any_page'


class Stage:
    def __init__(self, name, page_type, entry_point):
        self.name = name
        self.page_type = page_type
        self.entry_point = entry_point


# https://wiki.yandex-team.ru/wiki/b2b/kontent/eduwizzards/stages/
class Stages:
    PROMO = Stage('promo', PageType.ANY_PAGE, EntryPoint.VIEW)

    EDIT_BUTTON = Stage('edit-button', PageType.USER_CLUSTER, EntryPoint.VIEW)

    PAGE_MENU = Stage('page-menu', PageType.USER_CLUSTER, EntryPoint.VIEW)

    EDIT_FORMATTING = Stage('edit-formatting', PageType.USER_CLUSTER, EntryPoint.EDIT)

    EDIT_ATTACHING = Stage('edit-attaching', PageType.USER_CLUSTER, EntryPoint.EDIT)

    CREATE = Stage('create', PageType.ANY_PAGE, EntryPoint.CREATE)

    EDIT_BASICS = Stage('edit-basics', PageType.ANY_PAGE, EntryPoint.EDIT)

    ACCESS = Stage('access', PageType.ANY_PAGE, EntryPoint.ACCESS)

    SETTINGS = Stage('settings', PageType.ANY_PAGE, EntryPoint.SETTINGS)

    MOVE = Stage('move', PageType.ANY_PAGE, EntryPoint.MOVE)

    SUBSCRIBE = Stage('subscribe', PageType.ANY_PAGE, EntryPoint.SUBSCRIBE)

    USER_CLUSTER = Stage('user-cluster', PageType.ANY_PAGE, EntryPoint.VIEW)

    SUBSCRIBE_BUTTON = Stage('subscribe-button', PageType.ANY_PAGE, EntryPoint.VIEW)

    FAVORITES = Stage('favorites', PageType.ANY_PAGE, EntryPoint.VIEW)

    COMMENTS = Stage('comments', PageType.ANY_PAGE, EntryPoint.VIEW)

    SIDEBAR_REVISIONS = Stage('sidebar-revisions', PageType.ANY_PAGE, EntryPoint.VIEW)

    ACCESS_BUTTON = Stage('access-button', PageType.ANY_PAGE, EntryPoint.VIEW)

    EDIT_GRID = Stage('edit-grid', PageType.USER_GRID, EntryPoint.VIEW)

    EDIT_GRID_SIDEBAR = Stage('edit-grid-sidebar', PageType.USER_GRID, EntryPoint.VIEW)

    EDIT_GRID_ADD_COL = Stage('edit-grid-add-col', PageType.USER_GRID, EntryPoint.EDIT)

    EDIT_GRID_ADD_ROW = Stage('edit-grid-add-row', PageType.USER_GRID, EntryPoint.EDIT)


ALL_STAGES = (
    Stages.PROMO,
    Stages.EDIT_BUTTON,
    Stages.PAGE_MENU,
    Stages.EDIT_FORMATTING,
    Stages.EDIT_ATTACHING,
    Stages.CREATE,
    Stages.EDIT_BASICS,
    Stages.ACCESS,
    Stages.SETTINGS,
    Stages.MOVE,
    Stages.SUBSCRIBE,
    Stages.USER_CLUSTER,
    Stages.SUBSCRIBE_BUTTON,
    Stages.FAVORITES,
    Stages.COMMENTS,
    Stages.SIDEBAR_REVISIONS,
    Stages.ACCESS_BUTTON,
    Stages.EDIT_GRID,
    Stages.EDIT_GRID_SIDEBAR,
    Stages.EDIT_GRID_ADD_COL,
    Stages.EDIT_GRID_ADD_ROW,
)

NAME_TO_STAGE = {stage.name: stage for stage in ALL_STAGES}

ALL_STAGES_NAMES = {stage.name for stage in ALL_STAGES}

STAGE_TIMEOUTS = {
    (EntryPoint.VIEW, PageType.ANY_PAGE): 12 * 60 * 60,
}

DISABLE_TUTORIAL_TIMEOUT_WAFFLE_SWITCH_NAME = 'disable_tutorial_timeout'


class GetTutorialResult:
    def __init__(self, status, stage=None):
        self.status = status
        self.stage = stage

    def as_json(self):
        result = {'status': self.status}
        if self.stage:
            result['stage'] = self.stage.name
        return result

    class Status:
        DISPLAY_STAGE = 'display_stage'
        NOTHING_TO_DISPLAY = 'nothing_to_display'
        TUTORIAL_COMPLETED = 'tutorial_completed'


class NonexistentTutorialStage(Exception):
    pass


class NonexistentEntryPoint(Exception):
    pass


def get_next_tutorial_step(user, page, entry_point):
    if not settings.IS_BUSINESS:
        return GetTutorialResult(status=GetTutorialResult.Status.TUTORIAL_COMPLETED)

    if entry_point not in ALL_ENTRY_POINTS:
        raise NonexistentEntryPoint('No such entry point: "%s"' % entry_point)

    completed_stages = user.profile.get('tutorial', {})

    if not len(completed_stages):
        return GetTutorialResult(status=GetTutorialResult.Status.DISPLAY_STAGE, stage=Stages.PROMO)

    if len(completed_stages) == len(ALL_STAGES):
        return GetTutorialResult(status=GetTutorialResult.Status.TUTORIAL_COMPLETED)

    page_types = _determine_page_types(user, page)

    for page_type in page_types:
        for stage in ALL_STAGES:
            if _is_stage_suitable(stage, page_type, entry_point, completed_stages):
                return GetTutorialResult(status=GetTutorialResult.Status.DISPLAY_STAGE, stage=stage)

    return GetTutorialResult(status=GetTutorialResult.Status.NOTHING_TO_DISPLAY)


def set_tutorial_stage_completed(user, stage):
    if not settings.IS_BUSINESS:
        return

    if stage not in ALL_STAGES_NAMES:
        raise NonexistentTutorialStage('No such tutorial stage: "%s"' % stage)

    if 'tutorial' not in user.profile:
        _store_start_tutorial_state(user)

    user.profile['tutorial'][stage] = _get_current_timestamp_seconds()
    user.save()


def start_tutorial_again(user):
    if not settings.IS_BUSINESS:
        return None

    _store_start_tutorial_state(user)


def _is_stage_suitable(stage, page_type, entry_point, completed_stages):
    if stage.name in completed_stages:
        return False

    if stage.entry_point != entry_point or stage.page_type != page_type:
        return False

    timeout = _get_timeout(entry_point, page_type)

    if not timeout:
        return True

    filtered_completed_stages = {
        stage_name: stage_studied_timestamp
        for stage_name, stage_studied_timestamp in completed_stages.items()
        if (NAME_TO_STAGE[stage_name].entry_point == entry_point and NAME_TO_STAGE[stage_name].page_type == page_type)
    }

    stages_sorted_by_timestamp = sorted(
        iter(filtered_completed_stages.items()), key=operator.itemgetter(1), reverse=True
    )

    last_studied_stage = next(iter(stages_sorted_by_timestamp or []), None)

    if not last_studied_stage:
        return True

    stage_name, stage_studied_timestamp = last_studied_stage
    current_timestamp = _get_current_timestamp_seconds()

    return stage_studied_timestamp + timeout < current_timestamp


def _determine_page_types(user, page):
    if page.page_type == Page.TYPES.GRID and user in page.get_authors():
        return [PageType.USER_GRID]
    elif page.supertag == 'users/%s' % user.username:
        return [PageType.USER_CLUSTER, PageType.ANY_PAGE]
    return [PageType.ANY_PAGE]


def _store_start_tutorial_state(user):
    user.profile['tutorial'] = {}
    user.save()


def _get_timeout(entry_point, page_type):
    if waffle.switch_is_active(DISABLE_TUTORIAL_TIMEOUT_WAFFLE_SWITCH_NAME):
        return None
    return STAGE_TIMEOUTS.get((entry_point, page_type))


def _get_current_timestamp_seconds():
    return int(time.time())
