from typing import Union, List, Iterable, Optional

from django.contrib.auth import get_user_model
from django.core.exceptions import ObjectDoesNotExist

from wiki.api_frontend.serializers.user_identity import UserIdentity
from wiki.api_v2.public.pages.schemas import UserSchema
from wiki.sync.connect.injector import thread_locals

User = get_user_model()


class UserDataRepository:
    # мы можем не хранить ПД пользователя локально, а ходить за ними в источник (стафф, облако и т.д.)
    # и обеспечивать кеш вот тут

    def orm_to_user_schema(self, users: Union[User, Iterable[User]]) -> Union[User, List[User]]:
        if isinstance(users, Iterable):
            return [self._orm_to_user_schema(u) for u in users]
        return self._orm_to_user_schema(users)

    def _orm_to_user_schema(self, user: User) -> Optional[UserSchema]:
        if user is None:
            return None

        if not hasattr(thread_locals, '_user_repo_cache'):
            thread_locals._user_repo_cache = {}

        staff = thread_locals._user_repo_cache.get(user.id, None)
        if not staff:
            try:
                staff = user.staff
                thread_locals._staff_cache[user.id] = staff
            except (ObjectDoesNotExist, AttributeError):
                pass

        return UserSchema(
            id=user.id,
            identity=UserIdentity(uid=staff and int(staff.uid), cloud_uid=user.get_cloud_uid()),
            username=user.username,
            display_name=staff and staff.get_full_name(),
            is_dismissed=staff and staff.is_dismissed,
        )

    @classmethod
    def clear_cache(self, *args, **kwargs):
        """
        Очистить кэши
        *args, **kwargs нужны, ибо этот метод может быть повешен на сигнал
        """
        thread_locals._user_repo_cache = {}


USER_DATA_REPOSITORY = UserDataRepository()
