

from django.db.models import Q

from wiki.org import org_group, org_staff, org_user
from wiki.users_biz.models import GROUP_TYPES


def groups_by_ids(group_ids, group_type=GROUP_TYPES.group):
    """
    Вернуть список групп по переданным ID из Директории.

    @type group_ids: iterable
    """
    return org_group().filter(dir_id__in=group_ids, group_type=group_type)


def groups_by_department_ids(department_ids):
    """
    Вернуть департаментные группы по переданным id департаментов из Директории.

    @type department_ids: iterable
    """
    return groups_by_ids(department_ids, group_type=GROUP_TYPES.department)


def find_staff_models(q, limit):
    """
    Найти пользователей по строке q: если строка входит в login или first_name, или last_name.
    Достаются только поля id, uid, login, first_name, last_name.

    @type q: str
    @type limit: int
    @rtype: QuerySet
    """
    query = Q(user__is_active=True)
    for word in q.split():
        query &= Q(login__icontains=word) | Q(first_name__icontains=word) | Q(last_name__icontains=word)

    return (
        org_staff()
        .filter(query)
        .values('id', 'uid', 'login', 'first_name', 'last_name')
        .order_by(
            'last_name',
            'first_name',
            'login',
        )[:limit]
    )


def find_wiki_group_models(q, limit):
    """
    Найти группы по строке q: если строка входит в name.

    @type q: str
    @type limit: int
    @rtype: QuerySet
    """
    query = Q()
    for word in q.split():
        query &= Q(label__icontains=word) | Q(title__icontains=word)

    result = (
        org_group()
        .filter(query)
        .values('id', 'label', 'title')
        .order_by(
            'label',
        )[:limit]
    )
    # на фронте нужно значение title, но с именем name (для единообразия АПИ всех вик)
    return [{'id': obj['id'], 'name': obj['title']} for obj in result]


def get_admin_group_members():
    """
    Возвращает список пользователей, состоящих в группе аминистраторов
    """
    return org_user().filter(groups__group__group_dir_type='organization_admin')


def get_id_attr_name():
    """
    Вернуть имя атрибута в базе, который используется в качестве ID для данной сущности.
    """
    return 'dir_id'
