"""
Изначально сделано для https://st.yandex-team.ru/WIKI-10196.

Но может понадобиться и впоследствии.
"""

from time import sleep

from django.db import transaction
from requests.exceptions import ConnectionError

from wiki.sync.connect.models import Organization, ChangeEvent
from wiki.org import org_ctx, org_user
from wiki.sync.connect.dir_client import DirClientError, DirClient
from wiki.users.core import make_wiki_name
from wiki.users_biz.signals.dir_users import import_dir_user


def run(dir_org_ids=None, dry_run=True):
    messages = []
    error_types = {
        'broken_uid': 0,
        'broken_email': 0,
        'broken_is_active': 0,
        'no_such_in_db': 0,
        'no_such_in_dir': 0,
    }
    fix_errors = {'failed_to_import': 0}

    dir_client = DirClient(max_retries=10, timeout=30)

    def _repair_user_data(db_usr, dir_usr, org):
        id = dir_usr['id']
        email = dir_usr['email']
        nickname = dir_usr['nickname']
        first_name = dir_usr['name']['first'] or nickname
        last_name = dir_usr['name']['last'] or first_name
        gender = 'M' if dir_usr['gender'] == 'male' else 'F'
        is_robot = dir_usr.get('is_robot', False)
        service_slug = dir_usr.get('service_slug')
        is_active = not dir_usr['is_dismissed']

        db_usr.is_active = is_active
        db_usr.first_name = first_name
        db_usr.last_name = last_name
        db_usr.email = email
        db_usr.dir_id = id
        db_usr.is_dir_robot = is_robot
        db_usr.service_slug = service_slug
        db_usr.save()

        db_usr.staff.first_name_en = first_name
        db_usr.staff.last_name_en = last_name
        db_usr.staff.gender = gender
        db_usr.staff.lang_ui = org.lang
        db_usr.staff.is_dismissed = not is_active
        db_usr.staff.first_name = first_name
        db_usr.staff.last_name = last_name
        db_usr.staff.uid = id
        db_usr.staff.work_email = email
        db_usr.staff.wiki_name = make_wiki_name(db_usr)
        db_usr.staff.save()

    def _dismiss_user(db_usr):
        db_usr.is_active = False
        db_usr.save()

        db_usr.staff.is_dismissed = True
        db_usr.staff.save()

        for group in db_usr.groups.all():
            group.user_set.remove(db_usr)

    def _process_org(org):
        db_users = org_user()

        connection_errors = 0
        while True:
            try:
                dir_users, revision = dir_client.get_users(org.dir_id, is_dismissed='ignore')
                break
            except DirClientError as e:
                messages.append('dir_org_id={}, dir status {}'.format(org.dir_id, e.response.status_code))
                return
            except ConnectionError as e:
                connection_errors += 1
                timeout = 10
                print(
                    'Connection error #{} for org {}: {}; sleeping for {} seconds'.format(
                        connection_errors, org.dir_id, repr(e), timeout
                    )
                )
                sleep(timeout)

        login_to_db_user = {u.username: u for u in db_users}

        # Директория может вернуть несколько пользователей с одинаковым логином,
        # из которых в общем случае N могут быть уволенными и один активный.
        # Поскольку при увольнении и добавлении пользователя с тем же логином
        # мы переиспользуем БД запись пользователя с этим логином, то нам нужно
        # взять "последнего" пользователя с этим логином.
        login_to_dir_user = {u['nickname']: u for u in dir_users}

        for db_login, db_user in login_to_db_user.items():
            dir_user = login_to_dir_user.get(db_login)
            if dir_user:
                if str(dir_user['id']) != db_user.dir_id:
                    messages.append(
                        'dir_org_id={}, login={}, db_uid={}, dir_uid={}, repairing'.format(
                            org.dir_id, db_login, db_user.dir_id, dir_user['id']
                        )
                    )
                    error_types['broken_uid'] += 1
                    if not dry_run:
                        _repair_user_data(db_user, dir_user, org)
                elif dir_user['email'] != db_user.email:
                    messages.append(
                        'dir_org_id={}, login={}, db_email={}, dir_email={}, repairing'.format(
                            org.dir_id, db_login, db_user.email, dir_user['email']
                        )
                    )
                    error_types['broken_email'] += 1
                    if not dry_run:
                        _repair_user_data(db_user, dir_user, org)
                elif (not dir_user['is_dismissed']) != db_user.is_active:
                    messages.append(
                        'dir_org_id={}, login={}, db_is_active={}, dir_is_active={}, repairing'.format(
                            org.dir_id, db_login, db_user.is_active, not dir_user['is_dismissed']
                        )
                    )
                    error_types['broken_is_active'] += 1
                    if not dry_run:
                        _repair_user_data(db_user, dir_user, org)
            else:
                messages.append('dir_org_id={}, login={}, no such user in dir, dismissing'.format(org.dir_id, db_login))
                error_types['no_such_in_dir'] += 1
                if not dry_run:
                    _dismiss_user(db_user)

        logins_not_in_db = set(login_to_dir_user.keys()) - set(login_to_db_user.keys())
        for login_not_in_db in logins_not_in_db:
            dir_user = login_to_dir_user.get(login_not_in_db)
            messages.append('dir_org_id={}, login={}, no such user in DB, adding'.format(org.dir_id, login_not_in_db))
            error_types['no_such_in_db'] += 1
            if not dry_run:
                dir_user['org_id'] = org.dir_id
                try:
                    import_dir_user(sender=None, object=dir_user, content=None)
                except Exception:
                    fix_errors['failed_to_import'] += 1

    if dir_org_ids:
        orgs_qs = Organization.objects.filter(dir_id__in=dir_org_ids)
    else:
        orgs_qs = Organization.objects.all()

    for org in orgs_qs:
        with transaction.atomic():
            # Делаем SELECT FOR UPDATE, чтобы не пересечься с синхронизацией.
            ChangeEvent.objects.select_for_update().get(org__dir_id=org.dir_id)
            with org_ctx(org):
                _process_org(org)

    print('\n'.join(messages))
    print('\nError types:')
    for error_type, error_count in error_types.items():
        print('%s: %d' % (error_type, error_count))
    print('\nFix errors:')
    for error_type, error_count in fix_errors.items():
        print('%s: %d' % (error_type, error_count))
