from django.core.management.base import BaseCommand

from wiki.sync.connect.org_ctx import org_ctx, org_user
from wiki.sync.connect.dir_client import DirClient
from wiki.sync.connect.models import Organization
from wiki.users_biz.signals import import_dir_user


dir_client = DirClient()


class Command(BaseCommand):
    help = """
    Команды для импорта данных пользователей из Директории в базу Вики.
    Импортируются только те пользователи, которые отсутствуют в базе Вики или ID/CLOUD_UID которых передан явно.
    Если передан один параметр dir_org_id, то команда сравнивает список пользователей в Директории и в базе Вики
    и импортирует недостающих пользователей.
    Если передано два параметра: dir_org_id и dir_user_id/dir_user_cloud_uid, то команда импортирует только одного
    пользователя с переданным ID/CLOUD_UID.
    Если при этом такой пользователь уже существует в базе Вики, то его данные просто обновятся.
    """

    def add_arguments(self, parser):
        super(Command, self).add_arguments(parser)
        parser.add_argument('--dir_org_id', dest='dir_org_id', required=True, help='Organization ID in Directory')
        parser.add_argument('--dir_user_id', dest='dir_user_id', required=False, help='User ID in Directory')
        parser.add_argument(
            '--dir_user_cloud_uid',
            dest='dir_user_cloud_uid',
            required=False,
            help='Cloud UID of the User in the Directory',
        )

    def handle(self, *args, **options):
        dir_org_id = options.get('dir_org_id')
        dir_user_id = options.get('dir_user_id')
        dir_user_cloud_uid = options.get('dir_user_cloud_uid')
        if dir_user_id:
            import_user_directly(dir_org_id, dir_user_id)
        elif dir_user_cloud_uid:
            import_user_directly_by_cloud_uid(dir_org_id, dir_user_cloud_uid)
        else:
            import_missing_users_directly(dir_org_id)


def import_user_directly(dir_org_id, dir_user_id):
    org = Organization.objects.get(dir_id=dir_org_id)
    user = dir_client.get_user(org.dir_id, dir_user_id)
    user['org_id'] = org.dir_id
    with org_ctx(org):
        import_dir_user(sender=None, object=user, content=None)


def import_user_directly_by_cloud_uid(dir_org_id, dir_user_cloud_id):
    org = Organization.objects.get(dir_id=dir_org_id)
    user = dir_client.get_user_by_cloud_uid(org.dir_id, dir_user_cloud_id)
    user['org_id'] = org.dir_id
    with org_ctx(org):
        import_dir_user(sender=None, object=user, content=None)


def import_missing_users_directly(dir_org_id):
    org = Organization.objects.get(dir_id=dir_org_id)
    dir_users = dir_client.get_users(dir_org_id)[0]
    with org_ctx(org):
        wiki_users_dir_ids = [user.dir_id for user in org_user()]

    for dir_user in dir_users:
        if str(dir_user['id']) not in wiki_users_dir_ids:
            dir_user['org_id'] = org.dir_id
            print('Import user with dir_id=%s' % dir_user['id'])
            with org_ctx(org):
                import_dir_user(sender=None, object=dir_user, content=None)
