from typing import List, Optional

from django.contrib.auth.models import Group as DjangoGroup
from django.core.validators import MaxLengthValidator
from django.db import models
from django.utils.translation import ugettext as _

from wiki.acl.consts import AclSubject, AclSubjectType
from wiki.sync.connect.models import Organization
from wiki.org import get_org
from wiki.users.abstract_user import AbstractWikiUser
from .consts import DIRECTORY_GROUP_TYPES

GROUP_TYPES = DIRECTORY_GROUP_TYPES  # other places import this


def patch_validation_of_user_model_field(user_field, max_length):
    field = AbstractWikiUser._meta.get_field(user_field)

    field.max_length = max_length
    field.help_text = _('Required. %s characters or fewer. Letters, digits and @/./+/-/_ only.' % max_length)
    # убираем все валидаторы, кроме MaxLengthValidator с нужным значение max_length,
    # так как это мешает нам импортировать данные из Директории
    field.validators = [MaxLengthValidator(max_length)]


# https://st.yandex-team.ru/WIKI-9678
patch_validation_of_user_model_field('username', max_length=200)
patch_validation_of_user_model_field('first_name', max_length=200)
patch_validation_of_user_model_field('last_name', max_length=200)

# отключаем валидацию в поле email, так как это мешает нам импортировать данные из Директории
email_field = AbstractWikiUser._meta.get_field('email')
email_field.validators = list()


class User(AbstractWikiUser):
    """
    Модель внешнего Вики пользователя, использующаяся в инстансах Вики для Бизнеса.
    """

    def __str__(self):
        # Хак для того, чтобы в редактировании группы пользователь отображался по имени
        return '%s %s (%s)' % (self.first_name, self.last_name, self.username)

    # Идентификатор сущности в Директории
    dir_id = models.CharField(max_length=16, null=True, unique=True)
    cloud_uid = models.CharField(max_length=256, null=True, unique=True)

    # Все организации пользователя
    orgs = models.ManyToManyField(Organization, blank=True)

    is_dir_robot = models.BooleanField(default=False)

    service_slug = models.CharField(null=True, blank=True, db_index=True, max_length=200)

    @classmethod
    def find_by_cloud_uid(cls, cloud_uid):
        return cls.objects.get(cloud_uid=cloud_uid)

    def set_cloud_uid(self, cloud_uid):
        self.cloud_uid = cloud_uid

    def get_cloud_uid(self) -> Optional[str]:
        return self.cloud_uid

    def set_uid(self, uid: str):
        self.staff.uid = uid

    def get_uid(self) -> str:
        return self.staff.uid

    def get_all_groups(self):
        """
        Returns all groups the user is member of AND all parent groups of those groups.
        """

        def add_groups(result_groups, groups_to_add):
            for group in groups_to_add:
                result_groups.append(group.group)
                add_groups(result_groups, group.group.get_ancestors().all())

        result_groups = list()
        add_groups(result_groups, self.groups.filter(group__org=get_org()))

        return result_groups

    def get_all_group_ids(self) -> List[int]:
        return [group.id for group in self.get_all_groups()]

    class Meta(AbstractWikiUser.Meta):
        app_label = 'users_biz'


class Group(DjangoGroup):
    """
    Модель группы или департамента в Директории.
    Тип сущности задается атрибутом group_type.
    """

    # Идентификатор сущности в Директории
    dir_id = models.CharField(max_length=16)

    # Тип сущности: группа или департамент
    group_type = models.PositiveIntegerField(choices=GROUP_TYPES)

    title = models.CharField(max_length=200, null=True)

    label = models.CharField(max_length=200, db_index=True)

    # Тип группы в директории (у департаментов отсутствует), например, generic, organization_admin, department_head,
    group_dir_type = models.CharField(max_length=50, null=True)

    # Организация, в которой находится группа или департамент
    org = models.ForeignKey(Organization, null=True, blank=True, on_delete=models.CASCADE)

    parents = models.ManyToManyField('self', through='GroupRelationship', symmetrical=False, related_name='related_to')

    def get_public_group_id(self):
        return str(self.dir_id)

    def add_relation_to_parent(self, group):
        relationship, created = GroupRelationship.objects.get_or_create(
            from_group=self,
            to_group=group,
        )

        return relationship

    def remove_relation_to_parent(self, group):
        GroupRelationship.objects.filter(
            from_group=self,
            to_group=group,
        ).delete()

    def get_ancestors(self):
        """
        Вернуть все родительские группы, в которые непосредственно входит данная.
        """
        return self.parents.filter(to_group__from_group=self)

    def get_descendants(self):
        """
        Вернуть все дочерние группы, которые входят непосредственно в данную.
        """
        return self.related_to.filter(from_group__to_group=self)

    def get_all_members(self):
        """
        Вернуть список пользователей, которые входят в данную группу, как напрямую, так и через все дочерние группы
        на всех уровнях иерархии.
        """

        def add_users_from_groups(users, groups):
            for group in groups:
                users.update(set(group.user_set.all()))
                add_users_from_groups(users, group.get_descendants())

        users = set(self.user_set.all())
        add_users_from_groups(users, self.get_descendants())

        return list(users)

    def get_all_members_count(self):
        """
        Вернуть общее количество пользователей в данной группе и во всех дочерних группах на всех уровнях иерархии.
        """

        def get_members_count_in_groups(groups):
            members_count = 0
            for group in groups:
                members_count += group.user_set.count()
                members_count += get_members_count_in_groups(group.get_descendants())
            return members_count

        members_count = get_members_count_in_groups([self])

        return members_count

    def get_acl_subject(self):
        return AclSubject(subj_id=self.id, subj_type=AclSubjectType.GROUP)

    class Meta:
        db_table = 'auth_group_extra'
        unique_together = ('dir_id', 'group_type', 'org')
        app_label = 'users_biz'


class GroupRelationship(models.Model):
    """
    Модель определяет иерархию отношений между группами, между группами и департаментами и между департаментами.
    Одна группа может быть включена в родительскую группу и содержать в себе другие дочерние группы и департаменты.
    Один департамент может входит в состав родительского департамента или быть включен в какую-либо группу,
    а также включать в себя дочерние департаменты.
    """

    from_group = models.ForeignKey(Group, related_name='from_group', on_delete=models.CASCADE)
    to_group = models.ForeignKey(Group, related_name='to_group', on_delete=models.CASCADE)

    class Meta:
        db_table = 'auth_group_relationship'
        app_label = 'users_biz'
