from logging import getLogger

from django.dispatch import receiver

from wiki.sync.connect.org_ctx import get_org_dir_id
from wiki.sync.connect.signals import (
    department_added,
    department_deleted,
    department_moved,
    department_property_changed,
    group_added,
    group_deleted,
    group_membership_changed,
    group_property_changed,
)
from wiki.sync.connect.models import Organization
from wiki.org import org_ctx, org_group
from wiki.users_biz.errors import UnknownDirObjectError, UnknownDirUserError
from wiki.users_biz.models import GROUP_TYPES, Group
from wiki.users_biz.signals import get_dir_organization, get_dir_user

logger = getLogger(__name__)


@receiver(department_added)
def import_dir_department(sender, object, content, **kwargs):
    id = object['id']
    name = object['name']
    title = name
    label = object['label']
    dir_org_id = object['org_id']

    with org_ctx(_get_org(object)):
        parent = object.get('parent')
        if parent:
            # при первоначальном импорте у департамента в данных есть parent object
            parent_id = parent['id']
        else:
            # при синхронизации данных в событии есть только атрибут parent_id
            parent_id = object.get('parent_id')

        if org_group().filter(dir_id=id, group_type=GROUP_TYPES.department).exists():
            dep = get_dir_group_or_dep(id, GROUP_TYPES.department)
            if dep.title != title:
                dep.title = title
                dep.save()
            return

        org = get_dir_organization(dir_org_id)

        label_for_model, name_for_model = _get_label_and_name_for_group_model(dir_org_id, id, label, name)

        dep = Group.objects.create(
            dir_id=id,
            group_type=GROUP_TYPES.department,
            label=label_for_model,
            name=name_for_model,
            title=title,
            org=org,
        )

        logger.info(
            '\nNew department: id={id} dir_id={dir_id} label={label} title={title} parent_id={parent_id} '
            'dir_org_id={dir_org_id}'.format(
                id=dep.id,
                dir_id=id,
                label=label,
                title=title,
                parent_id=parent_id,
                dir_org_id=dir_org_id,
            )
        )

        if parent_id:
            parent_dep = get_dir_group_or_dep(parent_id, GROUP_TYPES.department)
            dep.add_relation_to_parent(parent_dep)


@receiver(department_moved)
def move_dir_department(sender, object, content, **kwargs):
    dep_dir_id = object['id']
    parent_ids = content['diff']['parent_id']

    with org_ctx(_get_org(object)):
        _execute_operation_for_groups_in_group(
            parent_ids[0], GROUP_TYPES.department, [dep_dir_id], GROUP_TYPES.department, 'remove_relation_to_parent'
        )

        _execute_operation_for_groups_in_group(
            parent_ids[1], GROUP_TYPES.department, [dep_dir_id], GROUP_TYPES.department, 'add_relation_to_parent'
        )


@receiver(department_property_changed)
def edit_dir_department(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        if 'diff' not in content:
            # TODO у некоторых событий отсутствует diff. Похоже на багу - https://st.yandex-team.ru/DIR-2346
            logger.warn('Секция "diff" не найдена в json события "department_property_changed"!')
            return
        _edit_group_or_dep(object['id'], content['diff'], GROUP_TYPES.department)


@receiver(group_added)
def import_dir_group(sender, object, content, **kwargs):
    id = object['id']
    name = object['name']
    title = name
    label = object['label']
    members = object['members']
    dir_type = object['type']
    dir_org_id = object['org_id']

    with org_ctx(_get_org(object)):
        if org_group().filter(dir_id=id, group_type=GROUP_TYPES.group).exists():
            group = get_dir_group_or_dep(id, GROUP_TYPES.group)
            if group.title != title:
                group.title = title
                group.save()
            return

        org = get_dir_organization(dir_org_id)

        label_for_model, name_for_model = _get_label_and_name_for_group_model(dir_org_id, id, label, name)

        group = Group.objects.create(
            dir_id=id,
            group_type=GROUP_TYPES.group,
            label=label_for_model,
            name=name_for_model,
            title=title,
            group_dir_type=dir_type,
            org=org,
        )

        logger.info(
            '\nNew group: id={id} dir_id={dir_id} label={label} title={title} '
            'type={type} dir_org_id={dir_org_id}'.format(
                id=group.id,
                dir_id=id,
                label=label,
                title=title,
                type=dir_type,
                dir_org_id=dir_org_id,
            )
        )

        for member in members:
            if member['type'] == GROUP_TYPES.department:
                dep = get_dir_group_or_dep(member['object']['id'], GROUP_TYPES.department)
                dep.add_relation_to_parent(group)
            elif member['type'] == 'user':
                # При первоначальном импорте данных организации, когда сначала импортируются группы, а потом
                # пользователи, мы игнорируем DoesNotExist при поиске пользователя, потому что его точно еще нет в базе,
                # а добавление пользователей в группы будет позже в обработчике события user_added
                try_to_import_nonexistent = sender and (
                    sender.__name__ == 'sync_dir_data_changes' or sender.__name__ == 'get_dir_group_or_dep'
                )
                user = get_dir_user(
                    member['object']['id'],
                    ignore_does_not_exist_error=sender and sender.__name__ == 'import_org_data',
                    try_to_import_nonexistent_user=try_to_import_nonexistent,
                )
                if user:
                    group.user_set.add(user)


@receiver(department_deleted)
def delete_dir_department(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        _delete_group_or_dep(object['id'], GROUP_TYPES.department)


@receiver(group_deleted)
def delete_dir_group(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        _delete_group_or_dep(object['id'], GROUP_TYPES.group)


@receiver(group_property_changed)
def edit_dir_group(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        _edit_group_or_dep(object['id'], content['diff'], GROUP_TYPES.group)


@receiver(group_membership_changed)
def edit_dir_group_membership(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        gr_id = object['id']

        try:
            _call_func_if_not_empty_list(gr_id, content['diff']['members']['add']['users'], _add_dir_users_to_group)
        except KeyError:
            pass
        try:
            _call_func_if_not_empty_list(
                gr_id, content['diff']['members']['remove']['users'], _remove_dir_users_from_group
            )
        except KeyError:
            pass
        try:
            _call_func_if_not_empty_list(
                gr_id, content['diff']['members']['add']['departments'], _add_dir_departments_to_group
            )
        except KeyError:
            pass
        try:
            _call_func_if_not_empty_list(
                gr_id, content['diff']['members']['remove']['departments'], _remove_dir_departments_from_group
            )
        except KeyError:
            pass
        try:
            _call_func_if_not_empty_list(gr_id, content['diff']['members']['add']['groups'], _add_dir_groups_to_group)
        except KeyError:
            pass
        try:
            _call_func_if_not_empty_list(
                gr_id, content['diff']['members']['remove']['groups'], _remove_dir_groups_from_group
            )
        except KeyError:
            pass


def get_dir_group_or_dep(dir_id, group_type):
    try:
        return org_group().get(dir_id=dir_id, group_type=group_type)
    except Group.DoesNotExist:
        # TODO: WIKI-10323 Убрать этот костыль
        # WIKI-10117, DIR-3050
        # Для некоторых видов групп/департаментов не создавались
        # события group_added/department_added, из-за этого ломалась
        # синхронизация, когда приходили события, связанные с такими
        # группами/департаментами. Для вновь создаваемых групп/департаментов
        # это будет исправлено, события всегда будут добавляться.
        # Но для старых групп/департаментов эти события в базе Директории
        # не появятся. Поэтому временно придется всякий раз, когда
        # мы не нашли группу/департамент в нашей базе, идти в /groups
        # и пытаться затянуть себе эту группу/департамент.
        # Костыль можно будет убрать, когда будет сделана DIR-3050,
        # и когда не останется незатянутых групп/департаментов.

        # >>>> Начало костыля
        from wiki.sync.connect.dir_client import DirClient
        from wiki.org import get_org

        dir_client = DirClient()
        if group_type == GROUP_TYPES.group:
            entity, _ = dir_client.get_groups(dir_org_id=get_org().dir_id, id=dir_id)
        else:
            entity, _ = dir_client.get_departments(dir_org_id=get_org().dir_id, id=dir_id)
        if entity:
            logger.warn('Lazy synced %s with id=%s', GROUP_TYPES[group_type], dir_id)
            entity = entity[0]
            entity['org_id'] = get_org().dir_id
            if group_type == GROUP_TYPES.group:
                import_dir_group(sender=get_dir_group_or_dep, object=entity, content=None)
            else:
                import_dir_department(sender=None, object=entity, content=None)
            return org_group().get(dir_id=dir_id, group_type=group_type)

        # <<<< Конец костыля

        msg = '%s with id=%s does not exist'
        logger.error(msg, GROUP_TYPES[group_type], dir_id)
        raise UnknownDirObjectError(msg % (GROUP_TYPES[group_type], dir_id))


def _add_dir_users_to_group(gr_dir_id, usr_dir_ids):
    execute_operation_for_users_in_group_or_dep(gr_dir_id, usr_dir_ids, GROUP_TYPES.group, 'add')


def _remove_dir_users_from_group(gr_dir_id, usr_dir_ids):
    execute_operation_for_users_in_group_or_dep(gr_dir_id, usr_dir_ids, GROUP_TYPES.group, 'remove')


def _add_dir_groups_to_group(parent_gr_dir_id, gr_dir_ids):
    _execute_operation_for_groups_in_group(
        parent_gr_dir_id, GROUP_TYPES.group, gr_dir_ids, GROUP_TYPES.group, 'add_relation_to_parent'
    )


def _remove_dir_groups_from_group(parent_gr_dir_id, gr_dir_ids):
    _execute_operation_for_groups_in_group(
        parent_gr_dir_id, GROUP_TYPES.group, gr_dir_ids, GROUP_TYPES.group, 'remove_relation_to_parent'
    )


def _add_dir_departments_to_group(parent_gr_dir_id, dep_dir_ids):
    _execute_operation_for_groups_in_group(
        parent_gr_dir_id, GROUP_TYPES.group, dep_dir_ids, GROUP_TYPES.department, 'add_relation_to_parent'
    )


def _remove_dir_departments_from_group(parent_gr_dir_id, dep_dir_ids):
    _execute_operation_for_groups_in_group(
        parent_gr_dir_id, GROUP_TYPES.group, dep_dir_ids, GROUP_TYPES.department, 'remove_relation_to_parent'
    )


def _edit_group_or_dep(dir_id, diff_data, group_type):
    group = get_dir_group_or_dep(dir_id, group_type)
    name_diff = diff_data.get('name')
    if group and name_diff and len(name_diff) > 1:
        # дифф имени выглядит так: {u'diff': {u'name': [u'old' u'new']}, где в списке сначала идет
        # предыдущее значение имени, а вторым элементом - новое, именно поэтому берем из списка второй элемент
        new_name = name_diff[1]
        if group.title != new_name:
            group.title = new_name
            group.save()


def _delete_group_or_dep(dir_id, group_type):
    try:
        group_or_dep = get_dir_group_or_dep(dir_id, group_type)
        if group_or_dep:
            group_or_dep.delete()
    except UnknownDirObjectError:
        logger.warn(
            'dir_org_id=%s, %s with dir_id=%s already deleted. Ignoring event %_deleted.',
            get_org_dir_id(),
            GROUP_TYPES[group_type],
            str(dir_id),
            GROUP_TYPES[group_type],
        )


def execute_operation_for_users_in_group_or_dep(gr_dir_id, usr_dir_ids, group_type, operation):
    # TODO: WIKI-10323
    # TODO: Убрать этот try/except одновременно с убиранием костыля из
    # TODO: get_dir_group_or_dep, после решения ситуации с WIKI-10117, DIR-3050,
    # TODO: вместо этого сортировать события из /events так, чтобы
    # TODO: group_membership_changed шло раньше group_deleted.
    # TODO: Так безопаснее, поскольку мы могли ошибочно не импортировать
    # TODO: группу/департамент, и в итоге проигнорировать события.
    #
    # Ситуация, когда группы/департамента уже нет при попытке обработки
    # добавления или исключения пользователей из группы/департамента,
    # возможна в двух случаях.
    #
    # 1)
    # При удалении группы из Директории, в рамках одной ревизии приходит
    # сначала событие group_deleted, а потом уже group_membership_changed
    # об удалении пользователей из этой группы. И на момент обработки
    # group_membership_changed группа уже удалена из нашей базы, и из
    # Директории, поэтому мы не можем ее достать. Но в этом случае можно
    # проигнорировать событие, т.к. группа и все ее relations уже удалены.
    #
    # 2)
    # (временная ситуация, в будущем ее не будет)
    # WIKI-10117, DIR-3050, WIKI-10273
    # Начало истории см. в обработчике except метода get_dir_group_or_dep.
    # Возможна ситуация, что group_added/department_added
    # не сгенерировалась, мы пошли в /groups или /departments, чтобы
    # создать группу/департамент, а группа/департамент уже удалена.
    # В этом случае ничего не остается, кроме как через try/catch
    # в обработчике события игнорировать такое событие.
    try:
        gr_or_dep = get_dir_group_or_dep(gr_dir_id, group_type)
    except UnknownDirObjectError:
        logger.info(
            'WIKI-10273: dir_org_id=%s, %s_id=%s: ' '%s already deleted, ignoring membership %s',
            get_org_dir_id(),
            GROUP_TYPES[group_type],
            str(gr_dir_id),
            GROUP_TYPES[group_type],
            operation,
        )
        return
    if gr_or_dep:
        for usr_dir_id in usr_dir_ids:
            try:
                usr = get_dir_user(str(usr_dir_id), try_to_import_nonexistent_user=True)
            except UnknownDirUserError:
                # Не нашли пользователя ни в базе Вики, ни в Коннекте через АПИ
                logger.info(f'User with dir_id={usr_dir_id} not found')
                continue
            getattr(gr_or_dep.user_set, operation)(usr)


def _execute_operation_for_groups_in_group(parent_gr_dir_id, parent_group_type, gr_dir_ids, group_type, operation):
    # TODO: WIKI-10323
    # TODO: Убрать этот try/except одновременно с убиранием костыля из get_dir_group_or_dep
    # TODO: get_dir_group_or_dep
    #
    # Далее try...except - это временный костыль для решения проблемы из тикета
    # https://st.yandex-team.ru/WIKI-10744#1520950264000.
    # После импорта всех старых организаций такая ситуация не должна повториться.
    try:
        parent_gr = get_dir_group_or_dep(parent_gr_dir_id, parent_group_type)
    except UnknownDirObjectError:
        logger.warn(
            'WIKI-10744: dir_org_id=%s, %s_id=%s: %s already deleted in Directory, ignoring membership %s',
            get_org_dir_id(),
            GROUP_TYPES[group_type],
            str(parent_gr_dir_id),
            GROUP_TYPES[group_type],
            operation,
        )
        return
    for gr_dir_id in gr_dir_ids:
        gr = get_dir_group_or_dep(gr_dir_id, group_type)
        if gr:
            getattr(gr, operation)(parent_gr)


def _get_label_and_name_for_group_model(dir_org_id, dir_group_id, label, name):
    label_for_model = label or name
    # Поле label имеет длину в базе 200 символов, поэтому обрезаем слишком длинное значение
    label_for_model = label_for_model[:200]

    # Стандартная Джанговая группа требует, чтобы name было уникальным, поэтому добавим к нему dir_org_id и
    # dir_group_id, но вместо name мы будем везде использовать label.
    # Получившееся значение name обрезаем по длине так, чтобы оно поместилось в поле name в базе, которое имеет
    # ограничение по длине 80 символов.
    name_for_model = '%s_%s_%s' % (label_for_model[:50], dir_org_id, dir_group_id)

    return label_for_model, name_for_model


def _call_func_if_not_empty_list(gr_id, obj_list, func):
    if obj_list:
        func(gr_id, obj_list)


def _get_org(object):
    return Organization.objects.get(dir_id=object['org_id'])
