from logging import getLogger

from django.conf import settings
from django.contrib.auth import get_user_model
from django.db import transaction
from django.dispatch import receiver
from ylog.context import log_context

from wiki.sync.connect.errors import DirSyncDataError
from wiki.sync.connect.signals import user_added, user_dismissed, user_moved, user_property_changed
from wiki.sync.connect.models import Organization
from wiki.intranet.models import Staff
from wiki.org import get_org, org_ctx
from wiki.pages.access.groups import get_wiki_service_group
from wiki.pages.models import Access, PageWatch
from wiki.personalisation.utils import create_user
from wiki.users.core import make_wiki_name
from wiki.users_biz.errors import UnknownDirObjectError, UnknownDirUserError
from wiki.users_biz.models import GROUP_TYPES
from wiki.users_biz.signals.dir_groups import (
    execute_operation_for_users_in_group_or_dep,
    get_dir_group_or_dep,
    get_dir_organization,
    get_dir_user,
)
from wiki.utils.lock import execute_with_lock

logger = getLogger(__name__)


@receiver(user_added)
def import_dir_user(sender, object, content, **kwargs):
    user_dir_id = object['id']
    execute_with_lock('dir_sync_user_%s' % user_dir_id, lambda: _import_dir_user_unsafe(object), timeout=30)


def _import_dir_user_unsafe(raw_user_dict):
    user_dir_id = raw_user_dict['id']
    email = raw_user_dict['email']
    nickname = raw_user_dict['nickname']
    # в форме эти поля обязательные. есть пользователи которые ставят " " и падают на валидаторе

    try:
        dir_first_name = raw_user_dict['name']['first'].strip() or nickname
        dir_last_name = raw_user_dict['name']['last'].strip()
    except KeyError:  # у роботов может не быть ['name']
        dir_first_name = nickname
        dir_last_name = None

    # у федеративных пользователей приходит
    # dir_first_name = "Имя Фамилия"
    # dir_last_name = "Имя Фамилия"

    # -------------------------------

    if not dir_last_name:
        if ' ' in dir_first_name:
            parts = dir_first_name.split(' ')
            dir_first_name = parts[0]
            dir_last_name = ' '.join(parts[1:])
        else:
            dir_last_name = dir_first_name

    # -------------------------------

    first_name = dir_first_name
    last_name = dir_last_name

    gender = 'M' if raw_user_dict.get('gender', 'male') == 'male' else 'F'

    department_id = raw_user_dict.get('department_id') or (
        raw_user_dict['department']['id'] if ('department' in raw_user_dict and raw_user_dict['department']) else None
    )
    groups = raw_user_dict.get('groups')
    dir_org_id = raw_user_dict['org_id']
    is_robot = raw_user_dict.get('is_robot', False)
    cloud_uid = raw_user_dict.get('cloud_uid')
    service_slug = raw_user_dict.get('service_slug')

    is_existing_user = False

    # может быть гонка, когда разные воркеры селери
    # обрабытвают параллельно импорт нескольких организаций
    # к которым принадлежит один пользователь

    with log_context(user_dir_id=user_dir_id, email=email, org_dir_id=dir_org_id), org_ctx(_get_org(raw_user_dict)):
        if get_user_model().objects.filter(dir_id=user_dir_id).exists():
            # WIKI-10046
            # Организация могла откатиться обратно из Коннекта в ПДД.
            # В этом случае организация удаляется из Коннекта, но остается
            # в нашей базе. Если организация вернется в Коннект, то мы снова
            # будем обрабатывать добавление пользователей с теми же uid, которые
            # уже есть в нашей базе. Но теперь этих пользователей надо добавить
            # в новую организацию.
            # Ситуация с существующим uid также может возникнуть, когда пользователь,
            # состоит в нескольких организациях.
            usr = get_dir_user(user_dir_id, ignore_does_not_exist_error=True)
            is_existing_user = True
        elif Staff.objects.filter(uid=user_dir_id).exists():
            # еще кейс неконсистентности, когда dir_id у пользователя пуст, хотя он есть в директории и синкнут
            st = Staff.objects.get(uid=user_dir_id)
            if st.user_id is None:
                raise DirSyncDataError(
                    'Staff model already exists for User, but no User is attached. Dir=%s' % user_dir_id
                )
            usr = st.user
            usr.dir_id = user_dir_id
            usr.save()
            return usr
        else:
            usr = create_user(
                username=nickname,
                email=email or '{}@noemail'.format(nickname),
                first_name=first_name,
                last_name=last_name,
                uid=user_dir_id,
            )

        if usr:
            logger.info(
                '\nNew user: id={id} dir_id={dir_id} username={username} email={email} '
                'first_name={first_name} last_name={last_name} dir_org_id={dir_org_id}'.format(
                    id=usr.id,
                    dir_id=str(user_dir_id),
                    username=nickname,
                    email=email,
                    first_name=first_name,
                    last_name=last_name,
                    dir_org_id=dir_org_id,
                )
            )

            org = get_dir_organization(dir_org_id)

            # Добавлять пользователя в организацию в случае
            # переиспользования логина не нужно, но будем
            # добавлять всегда, чтобы не плодить if-ы.

            usr.orgs.add(org)

            # В случае существования пользователя с указанным uid
            # и в случае переиспользования логина данные пользователя
            # могли измениться, поэтому перезаписываем атрибуты.

            if not is_robot or not is_existing_user:
                # WIKI-11972: не обновляем данные существующих в нашей базе роботов, так как это приводит к блокировкам
                # транзакций, запущенных на разных воркерах
                usr.is_active = True
                usr.first_name = first_name
                usr.last_name = last_name
                usr.email = email
                usr.dir_id = user_dir_id
                usr.is_dir_robot = is_robot
                usr.service_slug = service_slug
                usr.set_cloud_uid(cloud_uid)
                usr.save()

                usr.staff.is_dismissed = False
                usr.staff.first_name = first_name
                usr.staff.last_name = last_name
                usr.staff.uid = user_dir_id
                usr.staff.work_email = email
                usr.staff.wiki_name = make_wiki_name(usr)
                usr.staff.first_name_en = first_name
                usr.staff.last_name_en = last_name
                usr.staff.gender = gender
                usr.staff.lang_ui = org.lang
                usr.staff.save()

            if department_id:
                dep = get_dir_group_or_dep(department_id, GROUP_TYPES.department)
                dep.user_set.add(usr)
            if groups is not None:
                for group in groups:
                    try:
                        gr = get_dir_group_or_dep(group['id'], GROUP_TYPES.group)
                        gr.user_set.add(usr)
                    except UnknownDirObjectError:
                        if settings.DIRSYNC_STRICT_MODE:
                            raise
                        else:
                            logger.warning(
                                'No Group #%s in organization %s;'
                                'Since Strict Mode is off, skipping' % (group['id'], dir_org_id)
                            )
        else:
            raise DirSyncDataError(
                'Failed to create user dir_id=%d, email=%s, maybe it is not found on BB. '
                'Organization with dir_id=%s not imported' % (user_dir_id, email, str(dir_org_id))
            )


@receiver(user_dismissed)
def dismiss_dir_user(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        user = get_dir_user(object['id'], ignore_does_not_exist_error=True, try_to_import_nonexistent_user=True)
        if user is None:
            # если пользователя нет в нашей базе и невозможно импортнуть его через АПИ Директории, то пропускаем
            return

        try:
            department_id = object.get('department_id') or object['department']['id']
        except KeyError:
            department_id = None

        if department_id:
            execute_operation_for_users_in_group_or_dep(department_id, [user.dir_id], GROUP_TYPES.department, 'remove')

        with transaction.atomic():
            # удаляем пользователя из организации
            get_org().user_set.remove(user)

            # удаляем пользователя из доступов к страницам, принадлежащим данной организации
            Access.objects.filter(staff=user.staff, page__org=get_org()).select_related('page').delete()

            # удаляем пользователя из подписчиков к страницам организации
            PageWatch.objects.filter(user=user.username, page__org=get_org()).select_related('page').delete()

            if user.orgs.count() == 0:
                # если у пользователя больше нет организаций, то помечаем его как неактивный
                user.is_active = False
                user.save()

                user.staff.is_dismissed = True
                user.staff.save()


@receiver(user_property_changed)
def edit_dir_user(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        dir_user_id = object['id']
        try:
            user = get_dir_user(dir_user_id, try_to_import_nonexistent_user=True)
        except UnknownDirUserError:
            # если пользователя нет в нашей базе и невозможно импортнуть его через АПИ Директории, то пропускаем
            logger.info(f'User with dir_id={dir_user_id} not found')
            return

        is_changed = False

        role_diff = content['diff'].get('role')

        if role_diff:
            admin_group = get_wiki_service_group()
            if role_diff[0].lower() == 'user' and role_diff[1].lower() == 'admin':
                admin_group.user_set.add(user)
            elif role_diff[0].lower() == 'admin' and role_diff[1].lower() == 'user':
                admin_group.user_set.remove(user)
            else:
                with log_context(object=object, content=content):
                    raise ValueError('Unexpected role change combination during user_property_changed')

        gender_diff = content['diff'].get('gender')
        if gender_diff:
            user.staff.gender = 'M' if (gender_diff[1] and gender_diff[1].lower() == 'male') else 'F'
            is_changed = True

        name_diff = content['diff'].get('name')
        if name_diff and len(name_diff) > 1:
            # пример диффа имени:
            # [{u'last': u'wiki_test_user32', u'middle': u'', u'first': u'wiki_test_user32'},
            #  {u'last': u'wiki test_user33', u'middle': u'', u'first': u'wiki_test_user32'}]
            last = name_diff[1]['last']
            first = name_diff[1]['first']
            if last and user.last_name != last:
                user.last_name = last
                user.staff.last_name = last
                user.staff.last_name_en = last
                is_changed = True
            if first and user.first_name != first:
                user.first_name = first
                user.staff.first_name = first
                user.staff.first_name_en = first
                is_changed = True
        if is_changed:
            user.save()
            user.staff.save()


@receiver(user_moved)
def move_dir_user(sender, object, content, **kwargs):
    with org_ctx(_get_org(object)):
        user_dir_id = object['id']
        department_ids = content['diff']['department_id']

        execute_operation_for_users_in_group_or_dep(department_ids[0], [user_dir_id], GROUP_TYPES.department, 'remove')
        execute_operation_for_users_in_group_or_dep(department_ids[1], [user_dir_id], GROUP_TYPES.department, 'add')


def _get_org(object):
    return Organization.objects.get(dir_id=object['org_id'])
